/** @format */

import { services, paramsCheck, activityTimeCheck, registeInfos, userCheck } from '../decorator'
import { resultsModel } from '../sdk'
import { UserService, AccessService } from '../service'
import { formatVipCbUrl, getShopVip } from '../utils/package/vip'

export interface IVipParams {
  activityId: string
  callbackUrl?: string
  inviteId?: string // 需要助力回调时请求该接口
}
export interface ILoginParams {
  activityId: string
  userNick: string
  avatar: string
  isFollow: boolean
  inviteId?: string
}
export default class User {
  /**
   *  登录接口, 初始化/更新用户信息
   */
  @paramsCheck(['activityId', 'userNick', 'avatar', 'isFollow'])
  @activityTimeCheck
  @registeInfos(['userInfo', 'vipInfo', 'activityInfo'])
  @services([UserService, AccessService])
  async login(
    context: IContext<ILoginParams>,
    { userInfo, vipInfo, activityInfo }: IControllerInfos,
    [userService, accessService]: [UserService, AccessService]
  ): Promise<IResult<{ openId: string }>> {
    const { openId } = context

    if (!userInfo) {
      userInfo = await userService.initUserData(vipInfo, activityInfo)
    } else {
      userInfo = await userService.updateUserData(vipInfo, userInfo, activityInfo)
    }

    // access表
    await accessService.addAccess(userInfo)

    return resultsModel.success({
      openId
    })
  }
  /**
   *  获取会员信息
   */
  @paramsCheck(['activityId'])
  @registeInfos(['session'])
  @services([])
  async getVipInfo(
    context: IContext<IVipParams>,
    { session, activityUserNick }: IControllerInfos
  ): Promise<IResult<IVipInfo>> {
    const { callbackUrl } = context.data
    // 拼接回调参数
    const result = await getShopVip(context, session, callbackUrl || formatVipCbUrl(context), activityUserNick)

    return resultsModel.success(result)
  }

  /**
   *  获取排行榜
   */
  @paramsCheck(['activityId'])
  @userCheck({ intercept: true })
  @services([UserService])
  async getRankList(
    context: IContext<{
      activityId: string
    }>,
    { userInfo }: IControllerInfos,
    [userService]: [UserService]
  ) {
    const sortKey = 'score'
    const sortTime = 'updateScoreTime'

    const { userNick, avatar } = userInfo

    const rank = await userService.getMyRank(sortKey, sortTime, userInfo[sortKey])

    const list = await userService.getRank(sortKey, sortTime, 200)

    return resultsModel.success({
      userNick,
      avatar,
      rank,
      score: userInfo[sortKey],
      list
    })
  }
}
