/** @format */

import { CODE_TYPES, TASK_RATE_TYPE } from './../constants'
import { paramsCheck, userCheck, activityTimeCheck, noCatch, services, registeInfos } from '../decorator'

import { resultsModel, TBAPIS } from '../sdk'
import { TaskService, BaseService } from '../service'
import { logger } from '../utils'
import { ITaskInfo } from '../service/task.service'

export default class Task {
  /**
   * 任务列表
   */
  @paramsCheck(['activityId'])
  @userCheck({ intercept: true })
  @activityTimeCheck
  @registeInfos(['userInfo', 'vipInfo', 'activityInfo'])
  @services([TaskService])
  async getTaskList(
    context: IContext<{ activityId: string }>,
    { userInfo, vipInfo, activityInfo }: IControllerInfos,
    [taskService]: [TaskService]
  ): Promise<IResult<ITaskInfo[]>> {
    const { tasks, startTime } = activityInfo

    // 更新user状态
    let updatedUserInfo = await taskService.updateTasks(
      userInfo._id,
      // 需要查询更新的任务
      [
        taskService.updateVipTask(userInfo, vipInfo, tasks)
        // taskService.updateSignTask(userInfo, tasks),
        // await taskService.updateOrderGoodsTask(userInfo, tasks, startTime, session)
      ],
      userInfo
    )

    if (!updatedUserInfo) return resultsModel.error(CODE_TYPES.SYSTEM_ERROR, '更新任务列表失败')

    // 返回对应的任务时, B端b_activity_config tasks 需配置对应的任务

    return resultsModel.success([
      taskService.initTask('member', '加入会员', tasks, updatedUserInfo),
      taskService.initTask('follow', '关注店铺', tasks, updatedUserInfo)
      // taskService.initTask('sign', '签到', tasks, updatedUserInfo),
      // taskService.initTask('orderGoods', '购买商品', tasks, updatedUserInfo),
      // taskService.initTask('browseGoods', '浏览商品', tasks, updatedUserInfo),
      // taskService.initTask('collectGoods', '收藏商品', tasks, updatedUserInfo),
      // taskService.initTask('jumpLink', '浏览页面', tasks, updatedUserInfo),
      // taskService.initTask('invites', '邀请好友', tasks, updatedUserInfo),
      // taskService.initTask('exchangeCredits', '积分兑换', tasks, updatedUserInfo)
    ])
  }

  /**
   *  完成关注店铺任务
   */
  @paramsCheck(['activityId'])
  @activityTimeCheck
  @registeInfos(['userInfo', 'activityInfo'])
  @services([TaskService])
  async doFollowTask(
    context: IContext<{ activityId: string }>,
    { userInfo, activityInfo }: IControllerInfos,
    [TaskService]: [TaskService]
  ): Promise<IResult<boolean>> {
    let { follow } = userInfo

    const { flag, newFollow } = follow

    // 已经关注过店铺,不可完成
    const hasFollowed = flag || newFollow
    if (hasFollowed) return resultsModel.error(CODE_TYPES.ERROR_TASK_COMPLETE)

    // 更新状态
    const result = await TaskService.updateFollowTask(userInfo, activityInfo)
    return result ? resultsModel.success(true) : resultsModel.error(CODE_TYPES.SYSTEM_ERROR)
  }

  /**
   *  积分兑换次数
   */
  @paramsCheck(['activityId', 'credits'])
  @userCheck({ intercept: true })
  @activityTimeCheck
  @registeInfos(['userInfo', 'activityInfo', 'session'])
  @services([TaskService])
  async doExchangeCreditsTask(
    context: IContext<IParams>,
    { userInfo, activityInfo, session }: IControllerInfos,
    [taskService]: [TaskService]
  ): Promise<IResult<boolean>> {
    const { credits } = context.data
    const taskType = `exchangeCredits`
    const {
      tasks: {
        exchangeCredits: { value, times, taskRateType }
      }
    } = activityInfo

    const { todayCompleteTimes } = taskService.getTodayCompleteTask(taskType, userInfo)

    // 如果每日限制次数并到达了限制次数
    if (times && times === todayCompleteTimes && taskRateType === TASK_RATE_TYPE.EVERYDAY) {
      return resultsModel.error(CODE_TYPES.ERROR_TASK_LIMIT)
    }

    const myCredits = await TBAPIS.queryCredits(context, session)

    console.log('myCredits', myCredits)

    // 积分不足
    if (myCredits < credits) return resultsModel.error(CODE_TYPES.ERROR_NO_CREDITS)

    // 扣积分
    let result = await TBAPIS.changeCredits(context, session, {
      quantity: +credits,
      change_type: 2, //变更类型：0交易，1：互动活动，2：权益兑换，3：手工调整
      opt_type: 1, // 操作类型，'0'：增加，1：扣减
      remark: `积分兑换任务减积分`
    })

    if (!result) return resultsModel.error(CODE_TYPES.SYSTEM_ERROR)

    // 更新user表
    await taskService.updateUserTaskRecord(taskType, value, userInfo)

    return resultsModel.success(true)
  }

  /**
   * 浏览商品
   */
  @paramsCheck(['activityId', 'itemId'])
  @userCheck({ intercept: true })
  @activityTimeCheck
  @registeInfos(['userInfo', 'activityInfo'])
  @services([TaskService])
  async doBrowseGoodsTask(
    context: IContext<IParams>,
    { userInfo, activityInfo }: IControllerInfos,
    [taskService]: [TaskService]
  ): Promise<IResult<boolean>> {
    const taskType = 'browseGoods'
    const { itemId } = context.data
    const {
      tasks: {
        browseGoods: { taskRateType, times, itemIds, value }
      }
    } = activityInfo

    const itemIdArray: number[] = Array.isArray(itemIds) ? itemIds : itemIds.split(',').map(v => +v)

    // B端配置商品ids不匹配正确
    if (!itemIdArray.includes(+itemId)) return resultsModel.error(CODE_TYPES.PARAMS_ERROR, '商品id不匹配')

    const { todayTaskInfo, todayCompleteTimes } = taskService.getTodayCompleteTask('browseGoods', userInfo)

    const hasCompleteToday = todayTaskInfo.some(v => v.itemId === itemId) && taskRateType === TASK_RATE_TYPE.EVERYDAY

    // 今日任务已经完成
    if (hasCompleteToday) {
      return resultsModel.error(CODE_TYPES.PARAMS_ERROR, '今日该浏览商品任务已完成')
    }
    // 次数超限
    if (times && times === todayCompleteTimes && taskRateType === TASK_RATE_TYPE.EVERYDAY) {
      return resultsModel.error(CODE_TYPES.ERROR_TASK_LIMIT)
    }

    // 更新user表
    await taskService.updateUserTaskRecord(taskType, value, userInfo, {
      itemId
    })

    return resultsModel.success(true)
  }

  /**
   * 收藏商品
   */
  @paramsCheck(['activityId', 'itemId'])
  @userCheck({ intercept: true })
  @activityTimeCheck
  @registeInfos(['userInfo', 'activityInfo'])
  @services([TaskService])
  async doCollectGoodsTask(
    context: IContext<IParams>,
    { userInfo, activityInfo }: IControllerInfos,
    [taskService]: [TaskService]
  ): Promise<IResult<boolean>> {
    const taskType = 'collectGoods'
    const { itemId } = context.data
    const {
      tasks: {
        collectGoods: { taskRateType, times, itemIds, value }
      }
    } = activityInfo

    const itemIdArray: number[] = Array.isArray(itemIds) ? itemIds : itemIds.split(',').map(v => +v)

    // B端配置商品ids不匹配正确
    if (!itemIdArray.includes(+itemId)) return resultsModel.error(CODE_TYPES.PARAMS_ERROR, '商品id不匹配')

    const { todayTaskInfo, todayCompleteTimes } = taskService.getTodayCompleteTask(taskType, userInfo)

    const hasCompleteToday =
      todayTaskInfo.filter(v => v.itemId === itemId).length && taskRateType === TASK_RATE_TYPE.EVERYDAY

    // 今日任务已经完成
    if (hasCompleteToday) {
      return resultsModel.error(CODE_TYPES.PARAMS_ERROR, '今日该收藏商品任务已完成')
    }
    // 次数超限
    if (times && times === todayCompleteTimes && taskRateType === TASK_RATE_TYPE.EVERYDAY) {
      return resultsModel.error(CODE_TYPES.ERROR_TASK_LIMIT)
    }

    // 更新user表
    await taskService.updateUserTaskRecord(taskType, value, userInfo, {
      itemId
    })

    return resultsModel.success(true)
  }

  /**
   * 浏览链接
   */
  @paramsCheck(['activityId'])
  @userCheck({ intercept: true })
  @activityTimeCheck
  @registeInfos(['userInfo', 'activityInfo'])
  @services([TaskService])
  async doJumpLinkTask(
    context: IContext<IParams>,
    { userInfo, activityInfo }: IControllerInfos,
    [taskService]: [TaskService]
  ): Promise<IResult<boolean>> {
    const taskType = 'jumpLink'

    const {
      tasks: {
        jumpLink: { taskRateType, times, value, link }
      }
    } = activityInfo

    const { todayCompleteTimes } = taskService.getTodayCompleteTask(taskType, userInfo)

    const hasCompleteToday = times && todayCompleteTimes === times && taskRateType === TASK_RATE_TYPE.EVERYDAY

    // 今日任务已经完成
    if (hasCompleteToday) {
      return resultsModel.error(CODE_TYPES.PARAMS_ERROR, '今日跳转链接任务已完成')
    }
    // 次数超限
    if (times && times === todayCompleteTimes && taskRateType === TASK_RATE_TYPE.EVERYDAY) {
      return resultsModel.error(CODE_TYPES.ERROR_TASK_LIMIT)
    }

    // 更新user表
    await taskService.updateUserTaskRecord(taskType, value, userInfo)

    return resultsModel.success(true)
  }

  /**
   * 领取任务奖励
   */
  @paramsCheck(['activityId', 'taskType'])
  @userCheck({ intercept: true })
  @activityTimeCheck
  @registeInfos(['userInfo'])
  @services([TaskService])
  async receiveTaskRewards(
    context: IContext<IParams>,
    { userInfo }: IControllerInfos,
    [taskService]: [TaskService]
  ): Promise<IResult<{ rewards: number }>> {
    // 根据需求自定义领取任务增加的key值  如: gameTimes
    const awardsTargetKey = `gameTimes`

    const { taskType } = context.data

    const { remainTimes, _id } = userInfo

    // 暂无领取次数
    if (!remainTimes?.[taskType]) return resultsModel.error(CODE_TYPES.ERROR_TASK_NORECEIVE)

    const waitReceiveTimes = remainTimes?.[taskType]

    await taskService.updateUser(_id, {
      $set: {
        [`remainTimes.${taskType}`]: 0
      },
      $inc: {
        [awardsTargetKey]: waitReceiveTimes
      }
    })

    return resultsModel.success({
      rewards: waitReceiveTimes
    })
  }

  /**
   * 获取收藏商品列表
   */
  @paramsCheck(['activityId', 'itemIds'])
  @registeInfos(['userInfo', 'session'])
  @services([BaseService, TaskService])
  async getCollectGoodsList(
    context: IContext<IParams>,
    { session, userInfo }: IControllerInfos,
    [baseService, taskService]: [BaseService, TaskService]
  ) {
    const { itemIds } = context.data
    const list = await baseService.getItemListByIds(itemIds, session)

    const { taskInfo } = taskService.getCompleteTaskByUserTaskInfo('collectGoods', userInfo)

    const listWithCollectStatus = list.map(v => {
      return {
        ...v,
        // 完成列表是否含有itemId
        collected: taskInfo.some(completeItem => completeItem.itemId === v.itemId)
      }
    })

    return resultsModel.success(listWithCollectStatus)
  }
}
