/** @format */

import { USER_DB_NAME, ACCESS_DB_NAME, AWARDS_DB_NAME } from '../db'
import { resultsModel, dateFormatter, transformBeijingDate, BaseDao } from '../sdk'
import { CODE_TYPES } from '../constants'
import * as xlsx from 'node-xlsx'
import { services } from '../decorator/services'
import { StatService } from '../service'
import { registeInfos } from '../decorator'

export interface IGetStatsParams {
  activityId: string
  startDay: string
  endDay: string
}

export interface IAddStatsParams {
  activityId: string
  type: number
}

/** @format */
export default class StatController {
  /**
   *  增加埋点记录
   */
  @registeInfos(['userInfo'])
  @services([StatService])
  async addStat(context: IContext<IAddStatsParams>, { userInfo }: IControllerInfos, [statService]: [StatService]) {
    const { type } = context.data
    await statService.addStat(type, userInfo)
    return resultsModel.success(true)
  }
  /**
   *  统计数据
   */
  async getStats(context: IContext<IGetStatsParams>) {
    const { activityId, startDay, endDay } = context.data

    const userDao: IBaseDao = new BaseDao(context, USER_DB_NAME)
    const accessDao: IBaseDao = new BaseDao(context, ACCESS_DB_NAME)
    const oneDay = 24 * 3600 * 1000

    const xlsxData = []
    try {
      let day = startDay
      while (new Date(day).getTime() <= new Date(endDay).getTime()) {
        // 访问PV
        const PV = await accessDao.count({ activityId, createDay: day })
        // 访问UV
        const UV = (
          await accessDao.aggregate([
            { $match: { activityId, createDay: day } },
            { $group: { _id: '$openId', count: { $sum: 1 } } },
            { $count: 'uv' }
          ])
        )?.[0]?.uv
        // 新增UV
        const newUV = await userDao.count({
          activityId,
          createDay: day
        })

        // 已入会(老会员)PV
        const vipPV = await accessDao.count({
          activityId,
          createDay: day,
          //@ts-ignore
          $or: [{ 'member.flag': true }, { 'member.newMember': true }],
          'member.bememberDay': {
            $not: {
              $eq: day
            }
          }
        })

        // 已入会(老会员)UV
        const vipUV = (
          await accessDao.aggregate(
            [
              {
                $match: {
                  activityId,
                  createDay: day,
                  $or: [{ 'member.flag': true }, { 'member.newMember': true }],
                  'member.bememberDay': {
                    $not: {
                      $eq: day
                    }
                  }
                }
              },
              { $project: { openId: true } },
              { $group: { _id: '$openId', count: { $sum: 1 } } },
              { $count: 'uv' }
            ],
            { allowDiskUse: true }
          )
        )?.[0]?.uv

        // 未入会PV
        const noVipPV = await accessDao.count({ activityId, createDay: day, isVip: false })

        // 未入会UV
        const noVipUV = (
          await accessDao.aggregate([
            { $match: { activityId, createDay: day, isVip: false } },
            { $group: { _id: '$openId', count: { $sum: 1 } } }
          ])
        ).length

        // 新增入会UV
        const newVipUV = await userDao.count({
          activityId,
          'member.newMember': true,
          'member.bememberDay': day
        })

        const newFollowUV = await userDao.count({
          activityId,
          'follow.newFollow': true,
          'follow.followDay': day
        })

        // 根据任务类型获取完成任务的人数
        //  example: await getTaskCompleteUV('collectGoods', day)
        const getTaskCompleteUV = async (task: string, day: string) => {
          return await userDao.count({
            activityId,
            [`taskInfo.${day}.${task}`]: {
              $exists: true
            }
          })
        }

        const keyValueMapper = {
          时间: day,
          访问PV: PV,
          访问UV: UV,
          新增UV: newUV,
          已入会PV: vipPV,
          已入会UV: vipUV,
          未入会PV: noVipPV,
          未入会UV: noVipUV,
          新入会UV: newVipUV,
          新增关注UV: newFollowUV
          //收藏商品UV: await getTaskCompleteUV('collectGoods', day)
        }
        console.log(keyValueMapper, 'xlsxData')

        if (day === startDay) {
          xlsxData.push(Object.keys(keyValueMapper))
        }
        // @ts-ignore
        xlsxData.push(Object.values(keyValueMapper))
        let tommorrow = new Date(day).getTime() + oneDay
        day = dateFormatter(transformBeijingDate(tommorrow), 'yyyy/MM/dd')
      }

      let buffer = xlsx.build([
        {
          name: `数据统计${startDay.replace(/\//g, '-')}-${endDay.replace(/\//g, '-')}`,
          data: xlsxData
        }
      ])

      let result = await context.cloud.file.uploadFile({
        fileContent: buffer,
        fileName: `数据统计${startDay.replace(/\//g, '-')}-${endDay.replace(/\//g, '-')}.xlsx`
      })
      if (result.url) {
        result.url = result.url.replace('http', 'https').replace('-internal', '')
      }
      return resultsModel.success(result)
    } catch (e) {
      console.log('上传文件出错', e)
      // 打印日志
      return resultsModel.error(CODE_TYPES.SYSTEM_ERROR, `上传文件错误`)
    }
  }
}
