/** @format */

import { CODE_TYPES } from './../constants'
import { paramsCheck, userCheck, activityTimeCheck } from '../decorator/check'
import { services } from '../decorator/services'

import { resultsModel } from '../sdk'
import { getToday, setNewVipUserData, checkNewVip } from '../utils'
import { UserService, AccessService, TaskService, StatService } from '../service'
import { registeInfos } from '../decorator/registeInfos'
import { STAT_TYPE } from '../constants'

export default class Share {
  /**
   *  获取邀请信息
   */
  @paramsCheck(['activityId'])
  @userCheck({ intercept: true })
  @registeInfos(['userInfo', 'activityInfo'])
  @services([UserService])
  async getShareInfo(
    context: IContext<{
      activityId: string
    }>,
    { userInfo }: IControllerInfos
  ): Promise<
    IResult<{
      rewards: number
    }>
  > {
    const {
      remainTimes: { invites }
    } = userInfo

    return resultsModel.success({
      rewards: invites
    })
  }
  /**
   *  完成分享
   */
  @paramsCheck(['activityId', 'inviteId'])
  @activityTimeCheck
  @userCheck({ intercept: true })
  @registeInfos(['userInfo', 'vipInfo', 'activityInfo'])
  @services([UserService, TaskService, StatService, AccessService])
  async doHelp(
    context: IContext<{
      activityId: string
      inviteId: string
    }>,
    { userInfo, vipInfo, activityInfo }: IControllerInfos,
    [userService, taskService, statService]: [UserService, TaskService, StatService, AccessService]
  ): Promise<IResult<boolean>> {
    const { inviteId, activityId } = context.data
    const { openId } = context

    if (openId === inviteId) return resultsModel.error(CODE_TYPES.ERROR_NO_INVITE_SELF)

    const inviteUserInfo: IUserInfo = await userService.getUserInfoByOpenId(inviteId)

    // inviteId的用户信息查询不到, 则inviteId无效
    if (!inviteUserInfo) return resultsModel.error(CODE_TYPES.ERROR_INVALID_INVITE_ID)

    // 邀请成功有该用户信息
    const userInviteSuccessCount = await statService.getStatCountByquery({
      activityId,
      openId,
      type: STAT_TYPE.INITE_SUCCESS
    })
    if (userInviteSuccessCount) return resultsModel.error(CODE_TYPES.ERROR_AREADY_INVITE_SUCCESS)

    // 获取活动基本信息
    const { tasks } = activityInfo
    // 获取分享奖励
    const {
      invites: { value }
    }: ITasks = tasks

    // 新会员
    const isNewVip = checkNewVip(userInfo, vipInfo)

    // 新会员更新状态
    if (isNewVip && !userInfo.member.newMember) {
      await userService.updateUser(userInfo._id, {
        $set: {
          member: setNewVipUserData(userInfo.member)
        }
      })
    }

    // 满足邀请标准的条件, 根据需求变化
    const isValidShare = [isNewVip].every(v => !!v)

    const today = getToday()
    if (!isValidShare) return resultsModel.error(CODE_TYPES.ERROR_INVITE)

    // 成功邀请
    const { _id: inviteUserId } = inviteUserInfo
    // 增加邀请人的待领取次数和邀请记录
    await userService.updateUser(inviteUserId, {
      $inc: {
        'remainTimes.invites': value
      },
      $push: {
        [`taskInfo.${today}.invites`]: { openId, createTime: Date.now() }
      }
    })

    await statService.addStat(STAT_TYPE.INITE_SUCCESS, userInfo)

    return resultsModel.success(true)
  }
}
