/** @format */

import { services, paramsCheck, userCheck, activityTimeCheck } from '../decorator'
import { resultsModel } from '../sdk'
import { AwardsService } from '../service'
import { PRIZE_TYPE, DRAW_STATUS, CODE_TYPES } from '../constants'
import { getToday, logger } from '../utils'
import { CodeType } from 'taobao-mini-sdk/lib/utils/codetypes'

export interface IPrizeListParams {
  activityId: string
}

export interface IReceiveOjectParams {
  activityId: string
  id: string
  name: string
  phone: string
  province: string
  city: string
  area: string
  addressDetail: string
  streetName: string
}

export interface IReceiveEnameParams {
  activityId: string
  id: string
}

export default class AwardsController {
  /**
   *  我的奖品列表
   */
  @paramsCheck(['activityId'])
  @userCheck({ intercept: true })
  @services([AwardsService])
  async getMyPrizeList(
    context: IContext<IPrizeListParams>,
    {},
    [awardSer]: [AwardsService]
  ): Promise<IResult<{ list: IAwards[] }>> {
    const {
      openId,
      data: { activityId }
    } = context
    let result = await awardSer.getMyPrizeList({ openId, activityId })
    return resultsModel.success({
      list: result
    })
  }

  /**
   *  领取实物
   */
  @paramsCheck(['activityId', 'id', 'name', 'phone', 'province', 'city', 'area', 'addressDetail', 'streetName'])
  @userCheck({ intercept: true })
  @services([AwardsService])
  async receiveObjectPrize(
    context: IContext<IReceiveOjectParams>,
    {},
    [awardSer]: [AwardsService]
  ): Promise<IResult<boolean>> {
    let { province, city, area, streetName, addressDetail, id, name, phone, activityId } = context.data

    let result = await awardSer.recieveObjectPrize(id, {
      activityId,
      province,
      city,
      area,
      streetName,
      addressDetail,
      name,
      phone
    })

    if ((result as CodeType).code) {
      return resultsModel.error(result as CodeType)
    }
    return resultsModel.success(result as true)
  }

  /**
   *  权益重新领取
   */
  @paramsCheck(['activityId', 'id'])
  @userCheck({ intercept: true })
  @services([AwardsService])
  async receiveEnamePrize(context: IContext<IReceiveEnameParams>, {}, [awardSer]: [AwardsService]) {
    let { id, activityId } = context.data

    let result = await awardSer.recieveEnamePrize(id, activityId)
    if ((result as CodeType).code) {
      return resultsModel.error(result as CodeType)
    }
    if ((result as IAwards).remark) {
      return resultsModel.error(CODE_TYPES.SYSTEM_ERROR, (result as IAwards).remark)
    }
    return resultsModel.success(result)
  }

  /**
   *  概率抽奖
   */
  @paramsCheck(['activityId', 'prizeDataType'])
  @activityTimeCheck
  @services([AwardsService])
  async drawLotteryPrize(context: IContext<IParams>, { userInfo }: IControllerInfos, [awardSer]: [AwardsService]) {
    const { openId } = context
    const { activityId, prizeDataType } = context.data

    // 获取奖池配置
    const prizesPool = await awardSer.getPrizeConfig({
      activityId,
      // 根据需求配置筛选条件
      prizeDataType
    })

    const thanksPrize = prizesPool.find(v => v.type === PRIZE_TYPE.THANKS) || {
      type: PRIZE_TYPE.THANKS,
      prizeDataType,
      name: '谢谢参与'
    }

    // 根据概率获取
    let prize = await awardSer.getPrizeByProbability(prizesPool)

    // 未找到奖品,降级到谢谢参与
    if (!prize) {
      prize = thanksPrize
    }

    let reduceResult: ICodeType | number = 1

    // 不是积分奖品, 检查是否扣库存
    if (prize.type !== PRIZE_TYPE.CREDITS && prize.type !== PRIZE_TYPE.THANKS) {
      reduceResult = await awardSer.reduceStock(prize._id)
    }

    // 扣库存失败降级到谢谢参与
    if ((reduceResult as ICodeType)?.code || !reduceResult) {
      prize = thanksPrize
    }

    const { type, _id, ename, image, name, useUrl } = prize

    const { userNick } = userInfo
    let record = {
      openId,
      prizeId: _id,
      activityId,
      drawStatus: DRAW_STATUS.WAITAWARD,
      prizeDataType,
      remark: '',
      useUrl,
      type,
      ename,
      name,
      image,
      userNick,
      createDay: getToday()
    }

    // 奖品信息 insert c_awards表
    const result = await awardSer.addAward(record)

    return resultsModel.success({
      id: result,
      type,
      name,
      image
    })
  }
}
