/** @format */

import { checkParams, services, registeInfos, preCheck } from '../decorator/common'
import { resultsModel } from '../sdk'
import { CommonTaskService, CommonBaseService } from '../service/common'
import { ITaskInfo } from '../service/common/task.service'
import {
  checkActivityTime,
  checkTaskLimit,
  checkExchangeCreditsTask,
  checkRemainTimes,
  checkInviteId
} from '../utils/common/check/'
import { preUpdate } from '../decorator/common'
import { updateVip, updateSignTask, updateOrderGoods } from '../utils/common/update'
import { CODE_TYPES } from '../errorCode'

export default class Task {
  /**
   * 任务列表
   */
  @checkParams(['activityId', 'isVip'])
  @registeInfos(['vipInfo'])
  @services([CommonTaskService])
  @preCheck([checkActivityTime])
  @preUpdate([updateVip, updateSignTask, updateOrderGoods])
  async getTaskList(
    context: IContext<IParams>,
    { userInfo, activityInfo }: IControllerInfos,
    [taskService]: [CommonTaskService]
  ): Promise<IResult<{ list: Object[] }>> {
    // 读取B端 tasks字段配置自动生成任务列表
    const result = await taskService.initTaskList(userInfo, activityInfo)

    return resultsModel.success({
      ...result
    })
  }

  /**
   * 完成任务
   *
   * 商品相关任务 需要传itemId参数
   * 积分兑换任务 需要传credits参数
   */
  @checkParams(['activityId', 'key', 'itemId?', 'credits?', 'inviteId?', 'isVip?'])
  @registeInfos(['session'])
  @services([CommonTaskService])
  @preCheck([checkActivityTime, checkInviteId, checkTaskLimit, checkExchangeCreditsTask])
  async completeTask(
    context: IContext<IParams>,
    { userInfo, activityInfo }: IControllerInfos,
    [taskService]: [CommonTaskService],
    { inviteUserInfo }: IPreCheckData
  ) {
    const { key: taskType, isVip } = context.data

    // 更新user表
    const result = await taskService.completeTask(taskType, activityInfo, userInfo, inviteUserInfo)
    if((result as ICodeType)?.code){
      return resultsModel.error(result as ICodeType)
    }
    return resultsModel.success(result)
  }

  /**
   * 领取任务奖励
   * 领取的奖励增加的key值  如: gameTimes  前端传入
   */
  @checkParams(['activityId', 'key', 'rewardsKey'])
  @services([CommonTaskService])
  @preCheck([checkActivityTime, checkRemainTimes])
  async receiveTask(
    context: IContext<IParams>,
    { userInfo }: IControllerInfos,
    [taskService]: [CommonTaskService]
  ): Promise<IResult<{ rewards: number }>> {
    const { rewardsKey } = context.data
    const { key: taskType } = context.data

    const rewardsResult = await taskService.receiveTaskRewards(taskType, rewardsKey, userInfo)

    return resultsModel.success(rewardsResult)
  }

  /**
   * 获取收藏商品列表
   */
  @checkParams(['activityId', 'itemIds'])
  @registeInfos(['session'])
  @services([CommonBaseService, CommonTaskService])
  async getCollectGoods(
    context: IContext<IParams>,
    { session, userInfo }: IControllerInfos,
    [commonBaseService, commonTaskService]: [CommonBaseService, CommonTaskService]
  ) {
    const { itemIds } = context.data

    const list = await commonBaseService.getItemListByIds(itemIds, session)

    const listWithCollectStatus = await commonTaskService.getItemListWithCollectStatus(list, userInfo)

    return resultsModel.success(listWithCollectStatus)
  }
}
