/** @format */

import { getToday } from './getToday'
import { TASK_STATUS, TASK_RATE_TYPE } from '../../constants'
import { TASK_DB_NAME, OEDER_DB_NAME, TASK_RECORD_DN_NAME } from '../../db'
import { BaseDao } from '../../sdk'
import { dbFind, dbInsertOne } from './mongodb'

/**
 *
 * 获取任务今天完成的次数
 *
 * @param {string} taskType  任务类型
 * @param {IUserInfo} userInfo 用户信息
 * @returns {number} todayTimes  今日任务完成次数
 * @memberof TaskService
 */
export function getCompleteTaskTimes(
  taskType: string,
  userInfo: IUserInfo
): {
  todayCompleteTimes: number
  completeTimes: number
} {
  const today = getToday()
  const { taskInfo = {} } = userInfo
  let completeTimes = 0

  Object.keys(taskInfo).forEach(day => {
    if (taskInfo[day]?.[taskType]) {
      const dayTaskInfo = Number.isInteger(taskInfo[day]?.[taskType]) ? taskInfo[day]?.[taskType] : 0
      completeTimes += dayTaskInfo
    }
  })
  return {
    todayCompleteTimes: userInfo?.taskInfo?.[today]?.[taskType] || 0,
    completeTimes
  }
}

export async function getTotalCompleteTaskInfo(context: IContext<IParams>, taskType: string): Promise<ITaskRecord[]> {
  const {
    data: { activityId },
    openId
  } = context

  return await dbFind<ITaskRecord>(context, TASK_RECORD_DN_NAME, {
    activityId,
    openId,
    taskType
  })
}

export async function getTodayCompleteTaskInfo(context: IContext<IParams>, taskType: string): Promise<ITaskRecord[]> {
  const {
    data: { activityId },
    openId
  } = context

  return await dbFind<ITaskRecord>(context, TASK_RECORD_DN_NAME, {
    activityId,
    openId,
    taskType,
    createDay: getToday()
  })
}

export async function insertTaskRecord(
  context: IContext<IParams>,
  userInfo: IUserInfo,
  taskType: ITaskType,
  customRecord: object = {}
) {
  const {
    data: { activityId },
    openId
  } = context
  const now = Date.now()
  return await dbInsertOne<ITaskRecord>(context, TASK_RECORD_DN_NAME, {
    activityId,
    openId,
    taskType,
    createTime: now,
    updateTime: now,
    userNick: userInfo.userNick,
    createDay: getToday(),
    ...customRecord
  })
}

/**
 *
 *  根据用户和任务完成情况,设置任务状态
 *
 * @param {IUserInfo} useInfo
 * @param {string} taskType
 * @param {number} taskRateType
 * @param {number} [limitTimesEverday] 每天限制次数 任务频率为每天的时候必填
 * @return {taskTatus} 1未完成 2 待领取 3 已完成
 * @memberof TaskService
 */
export function setTaskStatus(
  useInfo: IUserInfo,
  taskType: string,
  taskRateType: number,
  limitTimesEveryday: number = 1
): number {
  const waitReceive = useInfo?.remainTimes?.[taskType] || 0
  if (waitReceive) return TASK_STATUS.WAIT_RECEIVE

  // 当天的完成次数 总共的完成次数
  const { todayCompleteTimes, completeTimes } = getCompleteTaskTimes(taskType, useInfo)

  switch (taskRateType) {
    case TASK_RATE_TYPE.FOREVER:
      return completeTimes > 0 ? TASK_STATUS.DONE : TASK_STATUS.WAIT_DO
      break
    case TASK_RATE_TYPE.EVERYDAY:
      return todayCompleteTimes >= limitTimesEveryday ? TASK_STATUS.DONE : TASK_STATUS.WAIT_DO
      break
    case TASK_RATE_TYPE.NOLIMIT:
      return TASK_STATUS.WAIT_DO
      break
  }
}

/**
 * 获取用户任务参与信息
 * @param context
 * @param openId
 * @param activityId
 */
export const getTask = async (context: IContext<IParams>, openId: string = '', activityId?: string): Promise<ITask> => {
  activityId = activityId ? activityId : context.data.activityId
  openId = openId ? openId : context.openId
  const taskInfodao: IBaseDao = new BaseDao(context, TASK_DB_NAME)
  const createDay = getToday()

  //查询每日任务完成记录
  const result: ITask = await taskInfodao.findOne({
    openId,
    activityId,
    createDay
  })

  //如果还没有今天的任务数据，则新增一条用户任务数据
  if (!result) {
    const insertObj = {
      openId,
      activityId,
      taskInfo: {},
      createTime: Date.now(),
      createDay: createDay
    }
    const insertResult = await taskInfodao.insertOne(insertObj)
    return { ...insertObj, _id: insertResult }
  }

  return result
}

/**
 * 获取养成类每日任务信息
 * @param taskType
 * @param task
 */
export function getTodayCompleteGrowTask(
  taskType: string,
  task: ITask
): {
  todayCompleteTimes: number
  todayTaskInfo: ITaskDetail[]
} {
  const today = getToday()
  return {
    todayCompleteTimes: task?.taskInfo?.[today]?.[taskType]?.length || task?.taskInfo?.[today]?.[taskType] || 0,
    todayTaskInfo: task?.taskInfo?.[today]?.[taskType] || []
  }
}

/**
 * 更新每日任务，适用于购买商品每日任务
 * @param context
 * @param activityId
 */
export async function updateTaskInfo(context: IContext<IParams>, _id: string, taskType: string, value) {
  let taskInfodao: IBaseDao = new BaseDao(context, TASK_DB_NAME)
  const today = getToday()
  return await taskInfodao.update(
    {
      _id
    },
    {
      $push: {
        [`taskInfo.${today}.${taskType}`]: value
      }
    }
  )
}

/**
 * 更新每日任务，适用于购买商品每日任务
 * @param context
 * @param activityId
 */
export async function getOrderCount(
  context: IContext<IParams>,
  openId: string,
  activityId: string,
  taskRateType: number
) {
  let orderdao: IBaseDao = new BaseDao(context, OEDER_DB_NAME)
  const project =
    taskRateType === TASK_RATE_TYPE.EVERYDAY
      ? {
          openId,
          activityId,
          createDay: getToday()
        }
      : {
          openId,
          activityId
        }

  return await orderdao.count(project)
}

/**
 *
 *  根据用户和任务完成情况,设置任务状态
 *
 * @param {IUserInfo} useInfo
 * @param {string} taskType
 * @param {number} taskRateType
 * @param {number} [limitTimesEverday] 每天限制次数 任务频率为每天的时候必填
 * @return {taskTatus} 1未完成 2 待领取 3 已完成
 * @memberof TaskService
 */
export function setGrowTaskStatus(
  userInfo: IUserInfo,
  taskType: string,
  taskRateType: number,
  limitTimesEveryday: number = 1,
  todayCompleteTimes: number,
  completeTimes: number
): number {
  const waitReceive = userInfo?.remainTimes?.[taskType] || 0
  if (waitReceive) return TASK_STATUS.WAIT_RECEIVE

  switch (taskRateType) {
    case TASK_RATE_TYPE.FOREVER:
      return completeTimes >= limitTimesEveryday ? TASK_STATUS.DONE : TASK_STATUS.WAIT_DO
      break
    case TASK_RATE_TYPE.EVERYDAY:
      return todayCompleteTimes >= limitTimesEveryday ? TASK_STATUS.DONE : TASK_STATUS.WAIT_DO
      break
    case TASK_RATE_TYPE.NOLIMIT:
      return TASK_STATUS.WAIT_DO
      break
  }
}
