/** @format */

import { resultsModel } from '../../../sdk'
import { CODE_TYPES } from '../../../errorCode'
import { getCompleteTaskTimes, getTodayCompleteTaskInfo } from '../task'
import { TASK_RATE_TYPE } from '../../../constants'
import { dbFind } from '../mongodb'
import { TASK_RECORD_DN_NAME } from '../../../db'
import { getToday } from '../getToday'

export default async function checkTaskLimit(
  context: IContext<{ activityId: string; taskType: ITaskType; itemId?: string }>,
  { userInfo, activityInfo }: IControllerInfos
) {
  const { taskType, itemId, activityId } = context.data
  const { openId } = context
  const { itemIds } = activityInfo?.tasks?.[taskType] || {}
  if (!taskType) {
    console.error('使用checkTaskLimit， 云函数必须传入taskType参数')
    return
  }
  if (!activityInfo?.tasks?.[taskType]) {
    return resultsModel.error(CODE_TYPES.SYSTEM_ERROR, `该活动缺少${taskType}配置项`)
  }

  if (itemId && !itemIds) {
    return resultsModel.error(CODE_TYPES.SYSTEM_ERROR, `该任务缺少商品配置项`)
  }

  // B端配置商品ids不匹配正确
  if (itemId && itemIds) {
    const itemIdArray: number[] = Array.isArray(itemIds) ? itemIds : itemIds.split(',').map(v => +v)
    if (!itemIdArray.includes(+itemId)) return resultsModel.error(CODE_TYPES.PARAMS_ERROR, '商品id不匹配')
  }

  const { taskRateType, times } = activityInfo?.tasks?.[taskType]

  // 永久任务已经完成
  const { completeTimes } = getCompleteTaskTimes(taskType, userInfo)
  const hasComplete = completeTimes && taskRateType === TASK_RATE_TYPE.FOREVER
  if (hasComplete) return resultsModel.error(CODE_TYPES.ERROR_TASK_LIMIT, `该任务已完成，去看看其它任务吧~`)

  // 每日任务今天已经完成
  const { todayCompleteTimes } = getCompleteTaskTimes(taskType, userInfo)
  const hasCompleteToday = times && todayCompleteTimes === times && taskRateType === TASK_RATE_TYPE.EVERYDAY
  if (hasCompleteToday) {
    return resultsModel.error(CODE_TYPES.ERROR_TASK_LIMIT, `该任务今天已完成，明天再来吧~`)
  }

  const todayTaskInfo = await getTodayCompleteTaskInfo(context, taskType)
  // 今日任务该商品已经完成
  const itemHasCompleteToday =
    todayTaskInfo.find(v => v.itemId && v.itemId === itemId) && taskRateType === TASK_RATE_TYPE.EVERYDAY

  if (itemHasCompleteToday) {
    return resultsModel.error(CODE_TYPES.PARAMS_ERROR, '今日该商品任务已完成')
  }
}
