/** @format */
import { CommonUserService, CommonBaseService } from '../../service/common'
import { getSellerSession, getUserOrderlist, getTask } from '../../utils'
import { formatVipCbUrl, getShopVip } from '../../utils/common/vip'
import { CODE_TYPES } from '../../errorCode'
import { resultsModel, TBAPIS } from '../../sdk'

export type IInfo = 'session' | 'vipInfo' | 'taobaoOrderist' | 'credits' | 'task'

async function initBaseInfo(context: IContext<IParams>, baseInfos: ICheckControllerInfos, needInfos: IInfo[]) {
  const handler = context?.cloud?.dataspace?.context?.handler

  return await getNeedInfos(context, baseInfos, needInfos, handler, [
    registeActivityInfo,
    registeUserInfo,
    registeSession,
    registeVipInfo,
    registeTaobaoOrderist,
    registeCredits,
    registeTask
  ])
}

async function registeActivityInfo(
  context: IContext<IParams>,
  baseInfos: ICheckControllerInfos,
  needInfos: IInfo[],
  handler: string
) {
  if (!baseInfos.activityInfo && handler !== 'getActivityBaseInfoById') {
    const baseService = new CommonBaseService(context)
    // 活动基本情况
    const activityInfo = await baseService.getBaseInfo(context.data.activityId)
    if (!activityInfo) return resultsModel.error(CODE_TYPES.ERROR_NO_ACTIVITY)
    baseInfos.activityInfo = activityInfo
  }
  return baseInfos
}

async function registeUserInfo(
  context: IContext<IParams>,
  baseInfos: ICheckControllerInfos,
  needInfos: IInfo[],
  handler: string
) {
  // 默认注入活动和用户信息并校验
  if (!baseInfos.userInfo) {
    const userService = new CommonUserService(context)
    const userInfo = await userService.getUserInfo()
    if (!userInfo && !baseInfos.noCheckUser) return resultsModel.error(CODE_TYPES.ERROR_NO_USER)
    baseInfos.userInfo = userInfo
  }

  return baseInfos
}

async function registeSession(
  context: IContext<IParams>,
  baseInfos: ICheckControllerInfos,
  needInfos: IInfo[],
  handler: string
) {
  if (needInfos.includes('session') && !baseInfos.session) {
    const { session } = await getSellerSession(context)
    baseInfos.session = session
  }
  return baseInfos
}

async function registeVipInfo(
  context: IContext<IParams>,
  baseInfos: ICheckControllerInfos,
  needInfos: IInfo[],
  handler: string
) {
  if (needInfos.includes('vipInfo') && !baseInfos.vipInfo && handler !== 'getVipInfo') {
    if (!baseInfos.session) {
      const { session, userNick } = await getSellerSession(context)
      baseInfos.session = session
    }
    const vipInfo = await getShopVip(context, baseInfos.activityInfo, formatVipCbUrl(context))
    baseInfos.vipInfo = vipInfo
  }
  return baseInfos
}

async function registeTaobaoOrderist(
  context: IContext<IParams>,
  baseInfos: ICheckControllerInfos,
  needInfos: IInfo[],
  handler: string
) {
  if (needInfos.includes('taobaoOrderist') && !baseInfos.taobaoOrderist) {
    const taobaoOrderist = await getUserOrderlist(
      context,
      baseInfos?.userInfo?.createTime || baseInfos?.activityInfo?.startTime || Date.now(),
      Date.now()
    )
    baseInfos.taobaoOrderist = taobaoOrderist
  }
  return baseInfos
}

async function registeCredits(
  context: IContext<IParams>,
  baseInfos: ICheckControllerInfos,
  needInfos: IInfo[],
  handler: string
) {
  if (needInfos.includes('credits') && !baseInfos.credits) {
    const credits = await TBAPIS.queryCredits(context)
    baseInfos.credits = credits
  }

  return baseInfos
}

async function registeTask(
  context: IContext<IParams>,
  baseInfos: ICheckControllerInfos,
  needInfos: IInfo[],
  handler: string
) {
  if (needInfos.includes('task') && !baseInfos.task) {
    const task = await getTask(context)
    if (!task._id) return resultsModel.error(CODE_TYPES.ERROR_UPDATE_RETRY)
    baseInfos.task = task
  }

  return baseInfos
}

type TRegisteFns = (
  context: IContext<IParams>,
  baseInfos: ICheckControllerInfos,
  needInfos: IInfo[],
  handler: string
) => Promise<ICheckControllerInfos | IErrorResult>

async function getNeedInfos(
  context: IContext<IParams>,
  baseInfos: ICheckControllerInfos,
  needInfos: IInfo[],
  handler: string,
  registeFns: TRegisteFns[]
) {
  let res: ICheckControllerInfos | IErrorResult = baseInfos
  for (let i = 0; i < registeFns.length; i++) {
    const registeFn = registeFns[i]
    res = await registeFn(context, baseInfos, needInfos, handler)

    if ((res as IErrorResult)?.code && !(res as IErrorResult)?.success) return res
  }
  return res
}

export default function registeInfos(needInfos: IInfo[]) {
  return function (target: any, name: string, descriptor: PropertyDescriptor) {
    const method = descriptor.value
    descriptor.value = async function (...args: any[]) {
      const [context, otherArgs = {}, services = [], preCheckData = {}] = args

      const baseInfo = await initBaseInfo(context, otherArgs, needInfos)

      if ((baseInfo as IErrorResult).code && !(baseInfo as IErrorResult).success) return baseInfo as IErrorResult

      return method.apply(target, [context, { ...baseInfo, registed: true }, services, preCheckData])
    }
  }
}
