/** @format */

import { resultsModel } from '../../../sdk'
import { CODE_TYPES } from '../../../errorCode'
import { formatVipCbUrl, getShopVip } from '../vip'
import { TOTAL_GUIDE_STEP } from '../../../constants'
import { getConfig } from '../../../config'
import { dbCount, dbFindOne } from '../mongodb'
import { INTERACT_TASK_DB_NAME } from '../../../db'
import { getToday } from '../getToday'

// 检查是否可以做活动任务
export async function checkInteractTaskValid(
  context: IContext<{
    activityId: string
    type: IGrowInteractTaskType
  }>
) {
  const { type, activityId } = context.data
  const { openId } = context

  const { INTERACT_TASKS } = getConfig(context)

  const interactTaskConfig = INTERACT_TASKS?.[type]
  if (!interactTaskConfig) return resultsModel.error(CODE_TYPES.ERROR_INVALID_INTERACT_TASK_TYPE)

  const { cdTime, dayLimit, totalLimit } = interactTaskConfig

  // 获取最新的任务记录
  const latestTaskInfo = await findLatestInteractTaskInfo(context, { activityId, type, openId })
  const now = Date.now()

  if (!latestTaskInfo) return {}

  const { createTime } = latestTaskInfo

  const gapTime = now - createTime

  // 任务冷却中
  const isCD = gapTime > 0 && gapTime <= cdTime
  if (isCD) return resultsModel.error(CODE_TYPES.ERROR_INVALID_CD_TIME)

  // 每日互动任务限制
  if (dayLimit) {
    const todayCount = await countTodayInteractTask(context, { activityId, type, openId })

    if (todayCount >= dayLimit) return resultsModel.error(CODE_TYPES.ERROR_INTERACT_TASK_DAY_LIMIT)
  }

  // 总次数限制
  if (totalLimit) {
    const totalCount = await countTotalInteractTask(context, { activityId, type, openId })

    if (totalCount >= totalLimit) return resultsModel.error(CODE_TYPES.ERROR_INTERACT_TASK_TOTAL_LIMIT)
  }
}

function findLatestInteractTaskInfo(
  context: IContext<IParams>,
  { activityId, type, openId }: { activityId: string; type: IGrowInteractTaskType; openId: string }
) {
  return dbFindOne<IDBInteractTasks>(
    context,
    INTERACT_TASK_DB_NAME,
    { activityId, openId, type },
    { sort: { createTime: -1 } }
  )
}

function countTodayInteractTask(
  context: IContext<IParams>,
  { activityId, type, openId }: { activityId: string; type: IGrowInteractTaskType; openId: string },
  day?: string
) {
  return dbCount(context, INTERACT_TASK_DB_NAME, { activityId, openId, type, createDay: day || getToday() })
}

function countTotalInteractTask(
  context: IContext<IParams>,
  { activityId, type, openId }: { activityId: string; type: IGrowInteractTaskType; openId: string },
  day?: string
) {
  return dbCount(context, INTERACT_TASK_DB_NAME, { activityId, openId, type })
}
