/** @format */

import { services, checkParams, preCheck, preUpdate } from '../decorator/common'
import { resultsModel } from '../sdk'
import { CommonUserService, CommonGameService } from '../service/common'
import {
  checkActivityTime,
  checkJoinId,
  checkUserInfo,
  checkVip,
  checkGameTimes,
  checkOpenPrizeStatus
} from '../utils/common/check'
import { updateUserInfo, reduceGameTimes } from '../utils/common/update'

export interface ISumitGameControllerInfos extends IControllerInfos {
  joinInfo: IJoinRecord
}

export default class Game {
  /**
   *  获取游戏信息
   *
   *  needKeys 根据需求自取字段  格式： { gameTimes: 1 }
   */
  @checkParams(['activityId', 'needKeys'])
  @services([CommonUserService])
  async getGameInfo(
    context: IContext<IParams>,
    { userInfo, activityInfo }: IControllerInfos,
    [userService]: [CommonUserService]
  ) {
    const { needKeys } = context.data
    const gameInfoResult = await userService.getUserInfo(needKeys)
    return resultsModel.success({
      ...gameInfoResult
    })
  }

  @checkParams(['activityId'])
  @services([CommonGameService])
  @preCheck([checkActivityTime, checkVip, checkGameTimes(3)])
  @preUpdate([reduceGameTimes(3)])
  async startGame(
    context: IContext<IParams>,
    { userInfo, activityInfo }: IControllerInfos,
    [gameService]: [CommonGameService]
  ) {
    const joinResult = await gameService.addJoinRecord(userInfo)

    return resultsModel.success({
      ...joinResult
    })
  }

  @checkParams(['activityId', 'id', 'score', 'rankKey?', 'timeKey?'])
  @services([CommonGameService])
  @preCheck([checkJoinId])
  async submitGame(
    context: IContext<IParams>,
    { userInfo }: ISumitGameControllerInfos,
    [gameService]: [CommonGameService],
    { joinInfo }: IPreCheckData
  ) {
    const { id, score, rankKey = 'totalScore', timeKey = 'updateScoreTime' } = context.data
    const result = await gameService.submitGame(id, score, rankKey, timeKey, userInfo, joinInfo)

    return resultsModel.success(result)
  }

  @checkParams(['activityId', 'rankKey?', 'timeKey?'])
  @services([CommonGameService])
  @preCheck([checkOpenPrizeStatus])
  async getMyRankPrize(
    context: IContext<IParams>,
    { userInfo, activityInfo }: IControllerInfos,
    [gameService]: [CommonGameService]
  ) {
    const { rankKey = 'totalScore', timeKey = 'updateScoreTime' } = context.data

    const myRankInfo = await gameService.getMyRankInfo(rankKey, timeKey, userInfo)

    return resultsModel.success({})
  }
}
