/**
 * 奖励
 */
const { BaseDao } = require('../sdk');
const UserService = require('./user.service');
const { CODE_TYPES, DRAW_STATUS, PRIZE_TYPE, ACTIVITY_STATUS } = require('../constants');
const {AWARDS_DB_NAME, PRIZE_CONFIG_DB_NAME } = require('../db');
const { sendTBAward, getSellerSession } = require('../utils');

class AwardsService extends UserService{
  constructor(context) {
    super(context);
    this.awardsdao = new BaseDao(context, AWARDS_DB_NAME);
    this.activityprizedao = new BaseDao(context, PRIZE_CONFIG_DB_NAME);
  }

  // 根据奖品Id扣库存, 扣库存成功返回为1
  async reduceStock(_id) {
    let result = await this.activityprizedao.findOne({_id, deleteStatus: ACTIVITY_STATUS.NORMAL});
    // 奖品不存在
    if (!result) {
      return CODE_TYPES.ERROR_NO_PRIZE;
    }
    let {switchStock, stock, useStock} = result;
    // 若不限制库存
    if (switchStock === 2) {
      return 1;
    }
    // 若库存不足
    if (useStock >= stock) {
      return CODE_TYPES.ERROR_NO_STOCK;
    }
    try{
      return await this.activityprizedao.update(
        {
          _id , 
          $where: "this.useStock < this.stock"
        }, 
        { 
          $inc: { 
            useStock: +1 
          }
        }
      );
    }catch(e) {
      console.log(`扣库存失败：`, e);
      return CODE_TYPES.SYSTEM_ERROR;
    }
  }

  /**
   * 插入奖品记录
   * @param {*} award 
   */
  async addAward(award) {
    try {
      return await this.awardsdao.insertOne({
        ...award,
        createTime: Date.now(),
        updateTime: Date.now()
      });
    }catch(e) {
      console.log(`插入中奖记录错误:`, e);
      return CODE_TYPES.SYSTEM_ERROR;
    }
  }

  /**
   * 我的奖品
   * @param {*} param0 
   */
  async getMyPrizeList({openId, activityId}) {
    // 获取奖品领取过期时间
    let { awardReceiveExpiredTime } = await this.getBaseInfo(activityId);
    let myprizeList = await this.awardsdao.find({openId, activityId}, {
      sort: {
        createTime: -1
      }
    });
    // 若有过期时间，且已过期
    if (awardReceiveExpiredTime) {
      let currentTime = Date.now()
      myprizeList.map(v => {
        if(currentTime > awardReceiveExpiredTime) {
          if ([DRAW_STATUS.WAITAWARD, DRAW_STATUS.RETRY].includes(v.drawStatus)) {
            v.drawStatus = DRAW_STATUS.EXPIRED;
            v.remark = `奖品已过期`;
          }
        }
        v.expiredTime = awardReceiveExpiredTime;
      })
    }
    return myprizeList;
  }

  // 领取实物
  async recieveObjectPrize(_id, {activityId, provice, city, area, addressdetail, name, phone}) {
    // 若有过期时间，且已过期
    let { awardReceiveExpiredTime } = await this.getBaseInfo(activityId);
    if (awardReceiveExpiredTime && Date.now() > awardReceiveExpiredTime) {
      return CODE_TYPES.ERROR_PRIZE_EXPIRED
    }
    let award = await this.awardsdao.findOne({_id});
    console.log(`receiveObject-award`, award);
    // 奖品不存在
    if (!award) {
      return CODE_TYPES.ERROR_NO_PRIZE;
    }
    // 已领取
    if (award.drawStatus === DRAW_STATUS.SUCCESS) {
      return CODE_TYPES.ERROR_RECEIVE_PRIZE;
    }
    // 不是实物，非法操作
    if (award.type !== PRIZE_TYPE.OBJECT) {
      return CODE_TYPES.ERROR_FORBIDDEN_OPE;
    }
    try {
      let result = await this.awardsdao.update({_id}, {
        $set: {
          receiveName: name,
          phone,
          drawStatus: DRAW_STATUS.SUCCESS,
          provice,
          city,
          area,
          addressdetail,
          receiveTime: Date.now(),
          updateTime: Date.now()
        }
      });
      return true;
    }catch(e) {
      console.log(`领取实物错误：`, e);
      return CODE_TYPES.SYSTEM_ERROR;
    }
  }

  // 发放淘宝权益（奖品数据已插入到awards_info表，且状态drawStatus 为1或者6）
  async recieveEnamePrize(_id, activityId) {
    // 若有过期时间，且已过期
    let { awardReceiveExpiredTime } = await this.getBaseInfo(activityId);
    if (awardReceiveExpiredTime && Date.now() > awardReceiveExpiredTime) {
      return CODE_TYPES.ERROR_PRIZE_EXPIRED;
    }
    let result = {};
    let award = await this.awardsdao.findOne({_id});
    // 奖品不存在
    if (!award) {
      return CODE_TYPES.ERROR_NO_PRIZE;
    }
    // 已领取
    if (award.drawStatus === DRAW_STATUS.SUCCESS) {
      return CODE_TYPES.ERROR_RECEIVE_PRIZE;
    }
    console.log(`award.drawStatus`, award.drawStatus);
    // 状态不是1,6
    if (![DRAW_STATUS.WAITAWARD, DRAW_STATUS.RETRY].includes(award.drawStatus)) {
      return CODE_TYPES.ERROR_FORBIDDEN_OPE;
    }
    let { session } = await getSellerSession(this.context);
    // 发放淘宝权益
    result = await sendTBAward(this.context, session, award);
    // 更新
    await this.awardsdao.update({_id}, {
      $set: {
        drawStatus: result.drawStatus,
        remark: result.remark,
        updateTime: Date.now()
      }
    });
    return result;
  }

  /**
   * 获取活动配置项奖品
   * @param {*} _id 
   */
  async getActivityPrizeById(_id) {
    return await this.activityprizedao.findOne({_id, deleteStatus: ACTIVITY_STATUS.NORMAL}, {
      sort: {
        index: 1
      }
    });
  }
}
module.exports = AwardsService;