/** @format */

import { getToday } from './getToday'
import { TASK_STATUS, TASK_RATE_TYPE } from '../../constants'

/**
 * 根据任务类型获取任务完成情况
 *
 * @param {IUserInfo} userInfo
 * @param {string} taskType
 * @returns {array}
 * @memberof TaskService
 */
export function getTotalCompleteTask(
  taskType: string,
  userInfo: IUserInfo
): {
  completeTimes: number
  taskInfo: ITaskDetail[]
} {
  const { taskInfo = {} } = userInfo
  let targetTaskList = []

  Object.keys(taskInfo).forEach(day => {
    if (taskInfo[day]?.[taskType]) {
      const dayTaskInfo = Array.isArray(taskInfo[day]?.[taskType])
        ? taskInfo[day]?.[taskType]
        : [taskInfo[day]?.[taskType]]
      targetTaskList = [...targetTaskList, ...dayTaskInfo]
    }
  })

  return {
    completeTimes: targetTaskList?.length || 0,
    taskInfo: targetTaskList
  }
}

/**
 *
 * 获取任务今天完成的次数
 *
 * @param {string} taskType  任务类型
 * @param {IUserInfo} userInfo 用户信息
 * @returns {number} todayTimes  今日任务完成次数  todayTaskInfo  今日对应任务完成详情
 * @memberof TaskService
 */
export function getTodayCompleteTask(
  taskType: string,
  userInfo: IUserInfo
): {
  todayCompleteTimes: number
  todayTaskInfo: ITaskDetail[]
} {
  const today = getToday()
  return {
    todayCompleteTimes: userInfo?.taskInfo?.[today]?.[taskType]?.length || 0,
    todayTaskInfo: userInfo?.taskInfo?.[today]?.[taskType] || []
  }
}

/**
 *
 *  根据用户和任务完成情况,设置任务状态
 *
 * @param {IUserInfo} useInfo
 * @param {string} taskType
 * @param {number} taskRateType
 * @param {number} [limitTimesEverday] 每天限制次数 任务频率为每天的时候必填
 * @return {taskTatus} 1未完成 2 待领取 3 已完成
 * @memberof TaskService
 */
export function setTaskStatus(
  useInfo: IUserInfo,
  taskType: string,
  taskRateType: number,
  limitTimesEveryday: number = 1
): number {
  const waitReceive = useInfo?.remainTimes?.[taskType] || 0
  if (waitReceive) return TASK_STATUS.WAIT_RECEIVE

  // 当天的完成次数
  const { todayCompleteTimes } = getTodayCompleteTask(taskType, useInfo)

  // 总共的完成次数
  const { completeTimes } = getTotalCompleteTask(taskType, useInfo)

  switch (taskRateType) {
    case TASK_RATE_TYPE.FOREVER:
      return completeTimes > 0 ? TASK_STATUS.DONE : TASK_STATUS.WAIT_DO
      break
    case TASK_RATE_TYPE.EVERYDAY:
      return todayCompleteTimes >= limitTimesEveryday ? TASK_STATUS.DONE : TASK_STATUS.WAIT_DO
      break
    case TASK_RATE_TYPE.NOLIMIT:
      return TASK_STATUS.WAIT_DO
      break
  }
}
