/**
 * 基本信息
 *
 * @format
 */

import { BaseDao, dateFormatter, transformBeijingDate } from '../../sdk'
import { STAT_DB_NAME, ACCESS_DB_NAME, USER_DB_NAME, AWARDS_DB_NAME, JOIN_DB_NAME, GITFT_CARD_DB_NAME } from '../../db'
import * as xlsx from 'node-xlsx'
import { getToday } from '../../utils'
import { PRIZE_DATA_TYPE, STAT_TYPE } from '../../constants'

export default class StatService {
  context: IContext<any>
  statdao: IBaseDao
  accessdao: IBaseDao
  userdao: IBaseDao
  awardsdao: IBaseDao
  joindao: IBaseDao
  giftdao: IBaseDao
  constructor(context: IContext<IParams>) {
    this.context = context
    this.statdao = new BaseDao(context, STAT_DB_NAME)
    this.accessdao = new BaseDao(context, ACCESS_DB_NAME)
    this.userdao = new BaseDao(context, USER_DB_NAME)
    this.joindao = new BaseDao(context, JOIN_DB_NAME)
    this.awardsdao = new BaseDao(context, AWARDS_DB_NAME)
    this.giftdao = new BaseDao(context, GITFT_CARD_DB_NAME)
  }

  /**
   *  添加数据统计记录
   *
   * @param {number} type  数据类型
   * @param {object} [customStatData={}] 自定义数据
   * @param {IUserInfo} [userInfo] 用户信息
   * @return {boolean}
   * @memberof StatService
   */
  async addStat(
    type: number,
    userInfo: IUserInfo = {} as IUserInfo,
    customStatData: object = {}
  ): Promise<{ id: string }> {
    const { openId } = this.context
    const { activityId } = this.context.data
    const { userNick } = userInfo
    const id = await this.statdao.insertOne({
      activityId,
      type,
      typeName: STAT_TYPE[type],
      userNick,
      openId,
      ...customStatData,
      createTime: Date.now(),
      createDay: getToday()
    })
    return { id }
  }

  async getStats(activityId: string, startDay: string, endDay: string): Promise<any> {
    let day = startDay
    const oneDay = 24 * 3600 * 1000
    let xlsxData = []
    let curDayTimestamp = new Date(day).getTime()
    const endDayTimestamp = new Date(endDay).getTime()
    while (curDayTimestamp <= endDayTimestamp) {
      // 访问PV
      const PV = await this.accessdao.count({ activityId, createDay: day })
      // 访问UV
      const UV = await this.userdao.count({ activityId, [`login.${day}`]: { $exists: true } })
      // 新增UV
      const newUV = await this.userdao.count({ activityId, createDay: day })
      // 新增PV
      // const newPV = await this.userdao.count({ activityId, createDay: day })

      // 抽卡PV
      const joinPV = await this.joindao.count({ activityId, createDay: day })

      const joinDetail = await this.joindao.aggregate([
        { $match: { activityId, createDay: day } },
        { $project: { openId: true } },
        { $group: { _id: '$openId', count: { $sum: 1 } } }
      ])
      // 抽卡UV
      const joinUV = joinDetail.length

      // 复参
      const repeatJoin = joinUV ? (joinPV / joinUV).toFixed(2) : '/'

      // 参与率
      const joinRate = joinUV ? (joinUV / UV).toFixed(4) : '/'

      // 新增UV(通过邀请)
      const newUVFromInviteUV = (
        await this.accessdao.aggregate([
          { $match: { activityId, createDay: day, inviteId: { $exists: true } } },
          { $project: { openId: true } },
          { $group: { _id: '$openId', count: { $sum: 1 } } }
        ])
      ).length

      // 已入会(老会员)PV
      const vipPV = await this.accessdao.count({
        activityId,
        createDay: day,
        //@ts-ignore
        $or: [{ 'member.flag': true }, { 'member.newMember': true }],
        'member.bememberDay': { $not: { $eq: day } }
      })

      // 已入会(老会员)UV
      const vipUV = await this.userdao.count({
        activityId,
        [`login.${day}`]: { $exists: true },
        $or: [
          //@ts-ignore
          { 'member.flag': true },
          //@ts-ignore
          { 'member.newMember': true, 'member.bememberTime': { $lt: new Date(day).getTime() } }
        ]
      })

      // 未入会PV
      const noVipPV = await this.accessdao.count({ activityId, createDay: day, isVip: false })

      // 未入会UV
      const noVipUV = (
        await this.accessdao.aggregate([
          { $match: { activityId, createDay: day, isVip: false } },
          { $project: { openId: true } },
          { $group: { _id: '$openId', count: { $sum: 1 } } }
        ])
      ).length

      // 新增入会UV
      const newVipUV = await this.userdao.count({ activityId, 'member.newMember': true, 'member.bememberDay': day })

      const newFollowUV = await this.userdao.count({ activityId, 'follow.newFollow': true, 'follow.followDay': day })

      // 助力成功UV
      const helpSuccessUV = (
        await this.statdao.aggregate([
          { $match: { activityId, createDay: day, type: STAT_TYPE.INITE_SUCCESS } },
          { $project: { openId: true } },
          { $group: { _id: '$openId', count: { $sum: 1 } } }
        ])
      ).length

      // 根据任务类型获取完成任务的人数
      //  example: await getTaskCompleteUV('collectGoods', day)
      const getTaskCompleteUV = async (task: string, day: string) => {
        return await this.userdao.count({
          activityId,
          [`taskInfo.${day}.${task}`]: { $exists: true }
        })
      }

      const getStatUVByType = async (type: number) => {
        return (
          await this.statdao.aggregate([
            { $match: { activityId, createDay: day, type } },
            { $project: { openId: true } },
            { $group: { _id: '$openId', count: { $sum: 1 } } }
          ])
        ).length
      }

      // 累计集齐1张卡的人数

      const getCardsUser = async (count: number) => {
        return await this.userdao.count({
          activityId,
          cardsCollectedCount: { $gte: count }
        })
      }

      const getJoinedTimes = async (times: number) => {
        return await this.userdao.count({
          activityId,
          joinedTimes: { $gte: times }
        })
      }

      const getJoinTimesAward = async (times: number, day: string) => {
        return await this.awardsdao.count({
          activityId,
          needTimes: times,
          prizeDataType: PRIZE_DATA_TYPE.JOIN_TIMES,
          createDay: day
        })
      }

      const keyValueMapper = {
        时间: day,
        访问PV: PV,
        访问UV: UV,
        新增活动访问UV: newUV,
        新增活动访问PV: await this.accessdao.count({ activityId, createDay: day }),
        抽卡PV: joinPV,
        抽卡UV: joinUV,
        复参: repeatJoin,
        参与率: joinRate,
        '访问UV（老会员）': vipUV,
        总新增入会数: newVipUV,
        点击我的卡牌UV: await getStatUVByType(STAT_TYPE.MY_CARD),
        点击赚次数UV: await getStatUVByType(STAT_TYPE.DO_TASK),
        发起邀请UV: await getStatUVByType(STAT_TYPE.INVITE),
        分享带来的访问UV: newUVFromInviteUV,
        助力页点击助力UV: await getStatUVByType(STAT_TYPE.HELP),
        '助力成功UV（即入会)': helpSuccessUV,
        助力后点击我也要赢大奖UV: await getStatUVByType(STAT_TYPE.HELP_WON_PRIZE),
        发起赠送卡片的UV: await getStatUVByType(STAT_TYPE.GIFT_CARD),
        赠送带来的访问UV: await getStatUVByType(STAT_TYPE.ACCESS_FROM_GIFT_CARD),
        好友领卡UV: (
          await this.giftdao.aggregate([
            { $match: { activityId, createDay: day, received: true } },
            { $project: { openId: true } },
            { $group: { _id: '$openId', count: { $sum: 1 } } }
          ])
        ).length,
        领卡后点击我也要赢大奖: await getStatUVByType(STAT_TYPE.GIFT_CARD_WON_PRIZE),
        已关注店铺UV: (
          await this.accessdao.aggregate([
            { $match: { activityId, createDay: day, $or: [{ 'follow.flag': true }, { 'follow.newFollow': true }] } },
            { $project: { openId: true } },
            { $group: { _id: '$openId', count: { $sum: 1 } } }
          ])
        ).length,
        通过活动关注店铺UV: newFollowUV,
        点击收藏商品UV: await getStatUVByType(STAT_TYPE.COLLECT_GOODS),
        成功收藏商品UV: await getTaskCompleteUV('collectGoods', day),
        点击去下单UV: await getStatUVByType(STAT_TYPE.ORDER),
        成功下单UV: (await getTaskCompleteUV('orderGoods', day)) + (await getTaskCompleteUV('orderDepositGoods', day)),
        总完成下单订单量: await getStatUVByType(STAT_TYPE.ORDER_RECORD),
        // '已入会(老会员)PV': vipPV,
        // 未入会PV: noVipPV,
        // 未入会UV: noVipUV,
        累计集齐1张卡的人数: await getCardsUser(1),
        累计集齐2张卡的人数: await getCardsUser(2),
        累计集齐3张卡的人数: await getCardsUser(3),
        累计集齐4张卡的人数: await getCardsUser(4),
        累计集齐5张卡的人数: await getCardsUser(5),
        累计集齐6张卡的人数: await getCardsUser(6),
        累计集齐7张卡的人数: await getCardsUser(7),
        累计集齐8张卡的人数: await getCardsUser(8),
        累计抽卡达到1次人数: await getJoinedTimes(1),
        累计抽卡达到10次人数: await getJoinedTimes(10),
        累计抽卡达到20次人数: await getJoinedTimes(20),
        累计抽卡达到50次人数: await getJoinedTimes(50),
        累计抽卡达到1次抽奖人数: await getJoinTimesAward(1, day),
        累计抽卡达到2次抽奖人数: await getJoinTimesAward(2, day),
        累计抽卡达到3次抽奖人数: await getJoinTimesAward(3, day),
        累计抽卡达到4次抽奖人数: await getJoinTimesAward(4, day)
      }
      console.log(keyValueMapper, 'xlsxData')

      if (day === startDay) {
        xlsxData.push(Object.keys(keyValueMapper))
      }
      // @ts-ignore
      xlsxData.push(Object.values(keyValueMapper))
      const tommorrow = new Date(day).getTime() + oneDay
      curDayTimestamp = tommorrow
      day = dateFormatter(transformBeijingDate(tommorrow), 'yyyy/MM/dd')
    }

    return xlsxData
  }

  async buildExcel(startDay: string, endDay: string, xlsxData: []) {
    let buffer = xlsx.build([
      {
        name: `数据统计${startDay.replace(/\//g, '-')}-${endDay.replace(/\//g, '-')}`,
        data: xlsxData
      }
    ])

    let result = await this.context.cloud.file.uploadFile({
      fileContent: buffer,
      fileName: `数据统计${startDay.replace(/\//g, '-')}-${endDay.replace(/\//g, '-')}.xlsx`
    })
    if (result.url) {
      result.url = result.url.replace('http', 'https').replace('-internal', '')
    }
    return result
  }

  async getStatCountByquery(query: IMongoQuery) {
    return await this.statdao.count(query)
  }
}
