/**
 * 基本信息
 *
 * @format
 */
import { Mixin } from 'ts-mixer'
import { BaseDao, resultsModel, TBAPIS } from '../../sdk'
import { GITFT_CARD_DB_NAME, JOIN_DB_NAME } from '../../db'
import { ACTIVITY_STATUS, CARD_PRIZE_STATUS, GIFT_CARD_STATUS, PRIZE_DATA_TYPE, PRIZE_TYPE } from '../../constants'
import { formatUserNick, getToday, logger } from '../../utils'
import UserService from './user.service'
import AwardService from './awards.service'
import { CODE_TYPES } from '../../errorCode'
import { uniq } from 'lodash'
import { setCardPrizeStatus } from '../../utils/custom/card'
import StatService from './stat.service'
export default class CardService extends Mixin(UserService, AwardService, StatService) {
  context: IContext<IParams>
  joindao: IBaseDao
  giftCarddao: IBaseDao
  constructor(context: IContext<IParams>) {
    super(context)
    this.joindao = new BaseDao(context, JOIN_DB_NAME)
    this.giftCarddao = new BaseDao(context, GITFT_CARD_DB_NAME)
  }

  // 集卡
  async collectCard(activityId: string, userInfo: IUserInfo) {
    const cardPool = await this.getPrizeConfig({
      activityId,
      prizeDataType: PRIZE_DATA_TYPE.CARD
    })

    const card = await this.getCard(cardPool)

    const { cardType, _id, image, name, type } = card

    const reduceResult = await this.reduceStock(_id)

    const today = getToday()

    if (type === PRIZE_TYPE.THANKS || (reduceResult as ICodeType)?.code) {
      await this.updateUser(userInfo._id, {
        $inc: {
          joinedTimes: 1,
          [`joinInfo.${today}`]: 1
        }
      })
      return {
        type: PRIZE_TYPE.THANKS,
        name: '谢谢参与'
      }
    }

    userInfo.cardInfo = userInfo.cardInfo || {}
    userInfo.cardInfo[cardType] = (userInfo.cardInfo?.[cardType] || 0) + 1

    const updateResult = await this.updateUser(userInfo._id, {
      $inc: {
        joinedTimes: 1,
        [`joinInfo.${today}`]: 1,
        [`cardInfo.${cardType}`]: 1
      },
      $set: {
        cardsCollectedCount: Object.keys(userInfo.cardInfo).length
      }
    })
    if (updateResult !== 1) return CODE_TYPES.SYSTEM_ERROR

    const myCardInfo = this.getMyCardInfo(userInfo, cardPool, cardType)

    const collectedCardTypePrizeList = await this.getCardPrizeList(
      activityId,
      PRIZE_DATA_TYPE.CARD_TYPE_AWARD,
      'needCards',
      myCardInfo.cardTypeCollectedCount
    )

    const unLockPrizeList = collectedCardTypePrizeList.filter(v => v.status === CARD_PRIZE_STATUS.UN_LOCK)
    return {
      cardType,
      image,
      name,
      type,
      ...myCardInfo,
      isNewCard: userInfo?.cardInfo?.[cardType] === 1,
      restCardTypeNextPrize:
        // @ts-ignore
        (collectedCardTypePrizeList.find(v => v.needCards > myCardInfo.cardTypeCollectedCount)?.needCards || 0) -
        myCardInfo.cardTypeCollectedCount,
      drawNeedCards: unLockPrizeList?.[0]?.needCards,
      drawLotteryStatus: unLockPrizeList.length ? CARD_PRIZE_STATUS.UN_LOCK : CARD_PRIZE_STATUS.LOCK
    }
  }

  getMyCardInfo(userInfo: IUserInfo, cardPool?: IActivityPrize[], cardType?: number) {
    const { cardInfo = {} } = userInfo
    // 该卡片收集数量
    const currCardCollectedCount = cardType ? cardInfo?.[cardType] || 0 : undefined

    // 已收集卡片的总数量
    const cardQuantityCollectedCount = Object.values(cardInfo).reduce((total, count) => total + count, 0)

    // 已收集卡片类型
    const cardTypeCollectedCount = Object.keys(cardInfo).length

    // 活动配置的卡片类型
    const cardTypeCount = cardPool ? cardPool.filter(v => v.type === PRIZE_TYPE.CARD).length : undefined
    return {
      currCardCollectedCount,
      cardQuantityCollectedCount,
      cardTypeCollectedCount,
      cardTypeCount,
      // 未收集的卡片类型
      restCardTypeCount: cardPool ? cardTypeCount - cardTypeCollectedCount : undefined
    }
  }

  async addCollectRecord(userInfo: IUserInfo, cardResult: object) {
    const { openId } = this.context
    const { activityId } = this.context.data
    const { userNick } = userInfo
    const recordId = await this.joindao.insertOne({
      activityId,
      userNick,
      openId,
      ...cardResult,
      createTime: Date.now(),
      createDay: getToday()
    })
    return { recordId }
  }

  // 获取卡片
  async getCard(cardPool: IActivityPrize[]) {
    let card = await this.getPrizeByProbability(cardPool)

    const thanksPrize = cardPool.find(v => v.type === PRIZE_TYPE.THANKS) || {
      type: PRIZE_TYPE.THANKS,
      prizeDataType: PRIZE_DATA_TYPE.CARD,
      name: '谢谢参与'
    }

    !card && (card = thanksPrize)

    return card
  }

  async getCardPrizeList(activityId: string, prizeDataType: number, needKey: string, userCount: number) {
    const { openId } = this.context

    const prizePool = await this.getPrizeConfig(
      {
        activityId,
        prizeDataType,
        type: { $ne: PRIZE_TYPE.THANKS }
      },
      {
        projection: { type: 1, image: 1, prizeDataType: 1, name: 1, [needKey]: 1, level: 1, isBackUp: 1 },
        sort: {
          level: 1
        }
      }
    )

    // 抽奖次数奖池分类 从小到大排列
    const category = uniq(prizePool.map(v => +v[needKey])).sort((a, b) => a - b)

    const prizeMapper = category.reduce((prev, curr) => {
      return {
        ...prev,
        [`${curr}`]: prizePool.filter(v => v[needKey] === curr).sort((a, b) => (a.isBackUp ? 1 : -1))
      }
    }, {})

    const awardList = await this.getAwardsInfoList(
      {
        activityId,
        openId,
        prizeDataType
      },
      {
        projection: { type: 1, image: 1, prizeDataType: 1, name: 1, needTimes: 1, needCards: 1 }
      }
    )

    console.log(prizeMapper)

    const prizeList = category.map(key => {
      return {
        [needKey]: key,
        status: setCardPrizeStatus(userCount, key, needKey, awardList),
        prizeList: [
          prizeMapper?.[key]?.[0],
          ...prizeMapper[key].filter((v: IActivityPrize, i: number) => {
            return !!v.isBackUp
          })
        ]
      }
    })

    return prizeList
  }

  async getAwardsCarouselList(activityId: string, limit: number = 30) {
    const list = await this.getAwardsInfoList(
      {
        activityId,
        prizeDataType: {
          $in: [PRIZE_DATA_TYPE.CARD_TYPE_AWARD, PRIZE_DATA_TYPE.JOIN_TIMES]
        },
        type: { $ne: PRIZE_TYPE.THANKS }
      },
      {
        sort: {
          createTime: -1
        },
        projection: {
          name: 1,
          image: 1,
          type: 1,
          prizeDataType: 1,
          avatar: 1,
          userNick: 1
        },
        limit
      }
    )

    return list.map(v => {
      return {
        ...v,
        userNick: formatUserNick(v.userNick)
      }
    })
  }
  async getMyCardsInfo(activityId: string, userInfo: IUserInfo) {
    const { cardInfo = {} } = userInfo
    const cardPool = await this.getPrizeConfig(
      {
        activityId,
        prizeDataType: PRIZE_DATA_TYPE.CARD
      },
      {
        projection: { name: 1, image: 1, cardType: 1 },
        sort: {
          level: 1
        }
      }
    )

    return cardPool.map(v => {
      const { name, image, cardType } = v
      return {
        type: cardType,
        name,
        image,
        count: cardInfo[cardType] || 0
      }
    })
  }
  /**
   *  生成赠卡记录
   *
   * @param {string} activityId
   * @param {number} type
   * @param {IUserInfo} userInfo
   * @memberof CardService
   */
  async doGiftCard(activityId: string, type: number, userInfo: IUserInfo) {
    const { openId, avatar, userNick } = userInfo
    const giftId = await this.giftCarddao.insertOne<IGiftCardRecord>({
      activityId,
      openId,
      type,
      userNick,
      avatar,
      received: false,
      createDay: getToday(),
      createTime: Date.now(),
      updateTime: Date.now()
    })

    if (!giftId) return CODE_TYPES.SYSTEM_ERROR
    return { giftId }
  }

  async getGiftCard(activityId: string, type: number, userInfo: IUserInfo, inviteUserInfo: IUserInfo) {
    // 扣除邀请者的对应卡牌
    const reduceResult = await this.userdao.update(
      {
        _id: inviteUserInfo._id,
        $where: `this.cardInfo[${type}] >= 1`
      },
      {
        $inc: { [`cardInfo.${type}`]: -1 }
      }
    )

    if (!reduceResult) return CODE_TYPES.SYSTEM_ERROR

    // 增加受赠者的卡牌
    const addResult = await this.updateUser(userInfo._id, {
      $inc: { [`cardInfo.${type}`]: 1 }
    })
    if (!addResult) return CODE_TYPES.SYSTEM_ERROR

    const cardInfo =
      (
        await this.getPrizeConfig({
          activityId,
          prizeDataType: PRIZE_DATA_TYPE.CARD,
          cardType: +type
        })
      )?.[0] || ({} as IActivityPrize)

    const { name, image } = cardInfo

    return {
      name,
      image,
      type
    }
  }

  async updateGiftCardRecord(_id: string, userInfo: IUserInfo) {
    const { openId: receiveOpenId, userNick: receiveUserNick, avatar: receiveAvatar } = userInfo
    const result = await this.giftCarddao.update(
      {
        _id,
        $where: `this.received === false`
      },
      {
        $set: {
          received: true,
          receiveOpenId,
          receiveUserNick,
          receiveAvatar,
          receiveTime: Date.now(),
          receiveDay: getToday()
        }
      }
    )

    if (result !== 1) return CODE_TYPES.SYSTEM_ERROR

    return result
  }
}
