/** @format */

import { checkParams, services, registeInfos, preCheck } from '../decorator/common'
import { resultsModel } from '../sdk'
import { CommonTaskService, CommonBaseService } from '../service/common'
import { ITaskInfo } from '../service/common/task.service'
import {
  checkActivityTime,
  checkTaskLimit,
  checkExchangeCreditsTask,
  checkRemainTimes,
  checkInviteId
} from '../utils/common/check/'
import { preUpdate } from '../decorator/common'
import { updateVip, updateSignTask, updateTaobaoGoods } from '../utils/common/update'

export default class Task {
  /**
   * 任务列表
   */
  @checkParams(['activityId'])
  @registeInfos(['vipInfo', 'taobaoOrderist'])
  @services([CommonTaskService])
  @preCheck([checkActivityTime])
  @preUpdate([updateVip, updateTaobaoGoods('orderGoods'), updateTaobaoGoods('orderDepositGoods')])
  async getTaskList(
    context: IContext<IParams>,
    { userInfo, activityInfo }: IControllerInfos,
    [taskService]: [CommonTaskService]
  ): Promise<IResult<{ list: ITaskInfo[] }>> {
    // 读取B端 tasks字段配置自动生成任务列表
    const result = await taskService.initTaskList(userInfo, activityInfo)

    return resultsModel.success({
      ...result
    })
  }

  /**
   * 完成任务
   *
   * 商品相关任务 需要传itemId参数
   * 积分兑换任务 需要传credits参数
   */
  @checkParams(['activityId', 'taskType', 'itemId?', 'credits?'])
  @services([CommonTaskService])
  @preCheck([checkActivityTime, checkTaskLimit, checkExchangeCreditsTask])
  async doCompleteTask(
    context: IContext<IParams>,
    { userInfo, activityInfo }: IControllerInfos,
    [taskService]: [CommonTaskService]
  ) {
    const { taskType } = context.data

    // 更新user表
    const result = await taskService.completeTask(taskType, activityInfo, userInfo)

    return resultsModel.success(result)
  }

  /**
   * 领取任务奖励
   * 领取的奖励增加的key值  如: gameTimes  前端传入
   */
  @checkParams(['activityId', 'taskType', 'rewardsKey'])
  @services([CommonTaskService])
  @preCheck([checkActivityTime, checkRemainTimes])
  async receiveTaskRewards(
    context: IContext<IParams>,
    { userInfo }: IControllerInfos,
    [taskService]: [CommonTaskService]
  ): Promise<IResult<{ rewards: number }>> {
    const { rewardsKey } = context.data
    const { taskType } = context.data

    const rewardsResult = await taskService.receiveTaskRewards(taskType, rewardsKey, userInfo)

    return resultsModel.success(rewardsResult)
  }

  /**
   * 获取收藏商品列表
   */
  @checkParams(['activityId', 'itemIds'])
  @registeInfos(['session'])
  @services([CommonBaseService, CommonTaskService])
  async getCollectGoodsList(
    context: IContext<IParams>,
    { session, userInfo }: IControllerInfos,
    [commonBaseService, commonTaskService]: [CommonBaseService, CommonTaskService]
  ) {
    const { itemIds } = context.data

    const list = await commonBaseService.getItemListByIds(itemIds)

    const listWithCollectStatus = await commonTaskService.getItemListWithCollectStatus(list, userInfo)

    return resultsModel.success({
      list: listWithCollectStatus
    })
  }
}
