/** @format */

import { services, checkParams, registeInfos, preUpdate } from '../decorator/common'
import { resultsModel } from '../sdk'
import { CommonAwardsService } from '../service/common'
import preCheck from '../decorator/common/preCheck'
import {
  checkActivityTime,
  checkPrizeExpired,
  checkValidEnamePrize,
  checkValidObjectPrize,
  checkCardPrize
} from '../utils/common/check/'
import { updateDrawRecord } from '../utils/common/update/updateUserInfo'

export interface IReceivePrizeControllerInfos extends IControllerInfos {
  awardInfo: IAwards
}

export default class AwardsController {
  /**
   *  我的奖品列表
   */
  @checkParams(['activityId'])
  @services([CommonAwardsService])
  async getMyPrizeList(
    context: IContext<IParams>,
    { activityInfo }: IControllerInfos,
    [awardSer]: [CommonAwardsService]
  ): Promise<IResult<{ list: IAwards[] }>> {
    const result = await awardSer.getMyPrizeList(activityInfo)
    return resultsModel.success(result)
  }

  /**
   *  领取实物
   */
  @checkParams(['activityId', 'id', 'name', 'phone', 'province', 'city', 'area', 'addressDetail', 'streetName'])
  @services([CommonAwardsService])
  @preCheck([checkPrizeExpired, checkValidObjectPrize])
  async receiveObjectPrize(
    context: IContext<IParams>,
    { activityInfo }: IReceivePrizeControllerInfos,
    [awardSer]: [CommonAwardsService]
  ) {
    let result = await awardSer.recieveObjectPrize(context)

    return resultsModel.success(result)
  }

  /**
   *  权益重新领取
   */
  @checkParams(['activityId', 'id'])
  @services([CommonAwardsService])
  @preCheck([checkPrizeExpired, checkValidEnamePrize])
  async receiveEnamePrize(
    context: IContext<IParams>,
    { session }: IReceivePrizeControllerInfos,
    [awardSer]: [CommonAwardsService],
    { awardInfo }: IPreCheckData
  ) {
    let { id } = context.data

    let result = await awardSer.recieveEnamePrize(id, awardInfo)

    return resultsModel.success(result)
  }

  /**
   * 概率抽奖
   * CARD_TYPE_AWARD: 7
   * JOIN_TIMES: 8
   * needCards: 需要卡牌的数量  prizeDataType=7存在
   * needTimes: 需要的抽奖次数 prizeDataType=8存在
   */
  @checkParams(['activityId', 'prizeDataType', 'needTimes?', 'needCards?'])
  @services([CommonAwardsService])
  @preCheck([checkActivityTime, checkCardPrize])
  @preUpdate([updateDrawRecord])
  async drawLotteryPrize(
    context: IContext<IParams>,
    { userInfo }: IControllerInfos,
    [awardSer]: [CommonAwardsService]
  ): Promise<
    IResult<{
      id?: string
      type: number
      name: string
      image?: string
    }>
  > {
    const { activityId, prizeDataType, needTimes, needCards } = context.data

    const drawResult = await awardSer.drawLottery(activityId, userInfo, {
      prizeDataType,
      needTimes,
      needCards
    })

    return resultsModel.success(drawResult)
  }
}
