/** @format */

import { BaseDao } from '../sdk'
import { ACTIVITY_CONFIG_DB_NAME, PRIZE_CONFIG_DB_NAME } from '../db'

import { DELETE_STATUS, OPEN_PRIZE_STATUS, PRIZE_TYPE, STOCK_TYPE } from '../constants'
import { signFigures } from '../utils'

export default class ActivityConfigService {
  activityconfigDao: IBaseDao
  activityconfigPrizeDao: IBaseDao
  constructor(context: Context) {
    this.activityconfigDao = new BaseDao(context, ACTIVITY_CONFIG_DB_NAME)
    this.activityconfigPrizeDao = new BaseDao(context, PRIZE_CONFIG_DB_NAME)
  }
  //保存新建活动
  async saveNewActivity(baseConfig: IActivityBaseInfo, prize: Array<IActivityPrize>, openId: string) {
    try {
      let result = await this.activityconfigDao.insertOne({
        ...baseConfig,
        createTime: Date.now(),
        updateTime: Date.now(),
        deleteStatus: DELETE_STATUS.EXIST,
        openPrizeStatus: OPEN_PRIZE_STATUS.WAIT_AWARD
      })
      console.log(result, '保存基础配置result')
      if (result) {
        try {
          prize = prize.map(item => {
            return {
              ...item,
              activityId: result,
              openId,
              createTime: Date.now(),
              updateTime: Date.now(),
              deleteStatus: DELETE_STATUS.EXIST
            }
          })
          return await this.activityconfigPrizeDao.insertMany(prize)
        } catch (e) {
          console.log(e, '保存奖品配置出错')
          // 如果奖品未保存删除之前的活动
          await this.activityconfigDao.delete({
            _id: result
          })
        }
      }
    } catch (e) {
      //保存活动失败
      console.log(e, '保存活动失败')
    }
  }

  // 更新活动基本信息
  async updateBaseConfigInfo(id: string, data: IActivityBaseInfo) {
    return await this.activityconfigDao.update(
      { _id: id },
      {
        $set: {
          ...data,
          updateTime: Date.now(),
          deleteStatus: DELETE_STATUS.EXIST
        }
      }
    )
  }

  // 通过id获取奖品信息
  async getPrizeInfoById(id) {
    let result = await this.activityconfigPrizeDao.find({
      _id: id
    })
    if (result && result[0]) {
      return result[0]
    }
    return false
  }

  // 奖品置为删除状态
  async updateDeleteStatusById(id) {
    return await this.activityconfigPrizeDao.update(
      {
        _id: id
      },
      {
        $set: {
          deleteStatus: DELETE_STATUS.DELETE
        }
      }
    )
  }

  async deletePrizeByIds(ids: string[]) {
    let result: any = true
    for (let i = 0; i < ids.length; i++) {
      result = await this.updateDeleteStatusById(ids[i])
    }
    return result
  }

  async insertAndUpdatePrize(activityId: string, allPrizeList: Array<IActivityPrize>, bothIds: string[]) {
    let result = {
      success: [],
      failed: []
    }
    for (let i = 0; i < allPrizeList.length; i++) {
      // 新奖品
      const isNewPrize = !allPrizeList[i]._id
      if (isNewPrize) {
        const insertResult = await this.insertPrizeByActivityId(activityId, allPrizeList[i])
        if (insertResult) {
          result.success.push(insertResult)
        } else {
          result.failed.push(insertResult)
        }
      }

      // 更新的奖品
      const isEditPrize = bothIds.includes(allPrizeList[i]._id)
      if (isEditPrize) {
        if (allPrizeList[i].switchStock === STOCK_TYPE.NO_LIMIT) {
          allPrizeList[i].useStock = 0
        } else if (allPrizeList[i].switchStock === STOCK_TYPE.LIMIT) {
          ;(allPrizeList[i].useStock || allPrizeList[i].useStock === 0) && delete allPrizeList[i].useStock
        }
        const editResult = await this.updateOnePrizeById(allPrizeList[i]._id, allPrizeList[i])
        if (editResult) {
          result.success.push(editResult)
        } else {
          result.failed.push(editResult)
        }
      }
    }
    return result
  }

  // 更新单条数据
  async updateOnePrizeById(id, data: IActivityPrize) {
    if (data.switchStock === STOCK_TYPE.NO_LIMIT) {
      data.useStock = 0
    } else {
      delete data.useStock
    }
    return await this.activityconfigPrizeDao.update(
      {
        _id: id
      },
      {
        $set: {
          ...data,
          updateTime: Date.now(),
          deleteStatus: DELETE_STATUS.EXIST
        }
      }
    )
  }

  // 插入单条奖品数据
  async insertPrizeByActivityId(activityId, data) {
    return await this.activityconfigPrizeDao.insertOne({
      ...data,
      activityId,
      useStock: 0,
      createTime: Date.now(),
      updateTime: Date.now(),
      deleteStatus: DELETE_STATUS.EXIST
    })
  }

  // 获取奖品所有配置
  async getAllOldPrizeList(activityId: string) {
    return await this.activityconfigPrizeDao.find({ activityId })
  }

  // 编辑活动时, 对比新旧奖品配置, 找出删除的id和更新的id
  async diffPrizeList(activityId: string, pirzeList: Array<IActivityPrize>) {
    const allNewIds = getIds(pirzeList)

    const oldPrizeList = await this.getAllOldPrizeList(activityId)

    const oldIds = getIds(oldPrizeList)

    let deleteIds: string[] = []
    let bothIds: string[] = oldIds.reduce((s, i) => {
      if (allNewIds.includes(i)) {
        s = [...s, i]
      } else {
        deleteIds.push(i)
      }
      return s
    }, [])

    return {
      deleteIds,
      bothIds
    }

    function getIds(pirzeList: Array<IActivityPrize>) {
      return pirzeList.reduce((s, v) => {
        if (v._id) {
          s = [...s, v._id]
        }
        return s
      }, [])
    }
  }

  //更新活动配置
  async updateActivity(activityId, data, prizeNoId) {
    console.log(activityId, data, prizeNoId, 'activityId, data, prizeNoId, oldPrizeIds')
    try {
      let originalData = await this.activityconfigDao.findOne({
        _id: activityId
      })
      let result = await this.activityconfigDao.update(
        {
          _id: activityId
        },
        {
          $set: {
            ...data,
            createTime: originalData.createTime,
            updateTime: Date.now()
          }
        }
      )
      if (result) {
        try {
          let prizeOldIdsByActivity = (
            await this.activityconfigPrizeDao.find({
              activityId
            })
          ).reduce((s, v) => {
            return (s = [...s, v._id])
          }, [])
          let prizeResult = await this.activityconfigPrizeDao.insertMany(prizeNoId)
          if (prizeResult) {
            //删除之前的奖品
            let deleteOldPrizeList = prizeOldIdsByActivity.reduce((s, item) => {
              return (s = [
                ...s,
                this.activityconfigPrizeDao.delete({
                  _id: item
                })
              ])
            }, [])
            return await Promise.all(deleteOldPrizeList)
          }
        } catch (e) {
          await this.activityconfigDao.update(
            {
              _id: activityId
            },
            {
              $set: {
                ...originalData[0]
              }
            }
          )
        }
      }
    } catch (e) {
      //更新失败
    }
  }

  //通过openId去获取当前商家活动列表
  async getActivityListByOpenId(openId: string, { pageSize = 10, pageNo = 1 }) {
    return {
      list: await this.activityconfigDao.find(
        {
          openId,
          deleteStatus: DELETE_STATUS.EXIST
        },
        {
          projection: {
            startTime: 1,
            endTime: 1,
            title: 1
          },
          sort: {
            createTime: -1
          },
          limit: pageSize,
          skip: (pageNo - 1) * pageSize
        }
      ),
      total: await this.activityconfigDao.count({
        openId,
        deleteStatus: DELETE_STATUS.EXIST
      }),
      pageSize,
      pageNo
    }
  }

  //通过activityId去获取活动配置信息
  async getActivityInfoByActivityId(activityId: string) {
    return {
      baseConfig: await this.activityconfigDao.findOne({
        _id: activityId,
        deleteStatus: DELETE_STATUS.EXIST
      }),
      prizeConfig: await this.activityconfigPrizeDao.find(
        { activityId, deleteStatus: DELETE_STATUS.EXIST, type: { $ne: PRIZE_TYPE.THANK } },
        { sort: { level: 1 } }
      )
    }
  }

  //通过activityId去删除活动
  async deleteActivityByActivityId(activityId: string) {
    // 删除活动基础配置
    let result = await this.activityconfigDao.update(
      { _id: activityId },
      { $set: { deleteStatus: DELETE_STATUS.DELETE } }
    )
    // 删除活动奖品配置
    if (result) {
      return await this.activityconfigPrizeDao.update({ activityId }, { $set: { deleteStatus: DELETE_STATUS.DELETE } })
    }
    return false
  }

  // 查询待开奖列表
  async getNodifyList(serverTime) {
    let list = await this.activityconfigDao.find({
      endTime: {
        $lt: serverTime
      },
      openPrizeStatus: {
        $in: [OPEN_PRIZE_STATUS.WAIT_AWARD, OPEN_PRIZE_STATUS.FAIL]
      },
      deleteStatus: DELETE_STATUS.EXIST
    })
    console.log(list)
    return list
  }

  // 更新openPrizeStatus为处理中
  async update2Process(_id) {
    try {
      await this.activityconfigDao.update(
        {
          _id
        },
        {
          $set: {
            openPrizeStatus: OPEN_PRIZE_STATUS.PROCESSING,
            updateTime: Date.now()
          }
        }
      )
      return true
    } catch (e) {
      // 日志记录
      console.log(e)
      return false
    }
  }

  // 更新openPrizeStatus为成功
  async update2Success(_id) {
    try {
      await this.activityconfigDao.update(
        {
          _id
        },
        {
          $set: {
            openPrizeStatus: OPEN_PRIZE_STATUS.SUCCESS,
            openPrizeMsg: '',
            updateTime: Date.now()
          }
        }
      )
      return true
    } catch (e) {
      // 日志记录
      console.log(e)
      return false
    }
  }

  // 更新openPrizeStatus为失败
  async update2Fail(_id, message) {
    try {
      await this.activityconfigDao.update(
        {
          _id
        },
        {
          $set: {
            openPrizeStatus: OPEN_PRIZE_STATUS.FAIL,
            openPrizeMsg: message || '',
            updateTime: Date.now()
          }
        }
      )
      return true
    } catch (e) {
      // 日志记录
      console.log(e)
      return false
    }
  }
  /**
   * 组装奖品数据
   *
   * @param {Array<IActivityPrize>} prizeConfig
   * @returns
   * @memberof ActivityConfigService
   */
  formatPrizeConfig(prizeConfig: Array<IActivityPrize>) {
    // level 根据level排序
    let prizeList = prizeConfig.sort((a, b) => a.level - b.level)
    // 风驰台 ejs渲染
    const poolConfig = [
      { conditionKeys: [], name: '抽卡规则', key: 'RateTable', sort: 'prizePool', prizeDataType: 6, customizeId: 6 },
      {
        conditionKeys: [{ key: 'needCards', value: '3' }],
        name: '解锁1张卡牌出奖规则',
        prizeDataType: 7,
        key: 'RatePrizePool2',
        sort: 'prizePool',
        customizeId: 7
      },
      {
        conditionKeys: [{ key: 'needCards', value: '5' }],
        name: '解锁2张卡牌出奖规则',
        prizeDataType: 7,
        key: 'RatePrizePool3',
        sort: 'prizePool',
        customizeId: 8
      },
      {
        conditionKeys: [{ key: 'needCards', value: '7' }],
        name: '解锁3张卡牌出奖规则',
        prizeDataType: 7,
        key: 'RatePrizePool4',
        sort: 'prizePool',
        customizeId: 9
      },
      {
        conditionKeys: [{ key: 'needTimes', value: '6' }],
        name: '累计抽卡1次出奖规则',
        prizeDataType: 8,
        key: 'RatePrizePool5',
        sort: 'prizePool',
        customizeId: 10
      },
      {
        conditionKeys: [{ key: 'needTimes', value: '10' }],
        name: '累计抽卡2次出奖规则',
        prizeDataType: 8,
        key: 'RatePrizePool6',
        sort: 'prizePool',
        customizeId: 11
      },
      {
        conditionKeys: [{ key: 'needTimes', value: '15' }],
        name: '累计抽卡3次出奖规则',
        prizeDataType: 8,
        key: 'RatePrizePool7',
        sort: 'prizePool',
        customizeId: 12
      },
      {
        conditionKeys: [{ key: 'needTimes', value: '20' }],
        name: '累计抽卡4次出奖规则',
        prizeDataType: 8,
        key: 'RatePrizePool8',
        sort: 'prizePool',
        customizeId: 13
      },
      {
        conditionKeys: [],
        name: '助力奖品',
        key: 'RatePrizePool9',
        sort: 'prizePool',
        prizeDataType: 9,
        customizeId: 14
      }
    ]
    // 取出奖池
    let prizeData = {}
    poolConfig.forEach(item => {
      if (item?.sort == 'prizePool') {
        prizeData[item.key] = prizeList.filter(prize => {
          return +item.prizeDataType == +prize.prizeDataType
        })
      }
    })
    // 过滤奖池条件
    poolConfig.forEach(item => {
      if (item?.sort == 'prizePool') {
        if (item.conditionKeys?.length) {
          item.conditionKeys.forEach(condition => {
            prizeData[item.key] = prizeData[item.key].filter(prize => {
              return prize[condition.key] == condition.value
            })
          })
        }
      }
    })
    return prizeData
  }
  // 概率奖品配置 properiodfrom  properiodto
  setPrizeProbalityRange(prizes: IActivityPrize[]) {
    let originPeriod = 1
    return prizes.map(v => {
      if (!v.probability) return v
      // 概率为0 跳出10000之外的区间
      if (+v.probability === 0)
        return {
          ...v,
          properiodfrom: 1000000,
          properiodto: 1000000
        }
      const properiodfrom = originPeriod
      const properiodto = +originPeriod + signFigures(+v.probability * 100) - 1
      originPeriod = +originPeriod + signFigures(+v.probability * 100)
      return {
        ...v,
        properiodfrom,
        properiodto
      }
    })
  }
}
