/** @format */

import { CODE_TYPES, EIGHT_HOURS, PRIZE_TYPE, SHIP_STATUS } from './../constants'
import { services, paramsCheck } from '../decorator'

import { resultsModel } from '../sdk'
import { AwardsService } from '../service'
import { formatTime } from '../utils/index'
import * as xlsx from 'node-xlsx'

export default class BaseController {
  /**
   *  活动中奖名单
   */
  @paramsCheck(['activityId'])
  @services([AwardsService])
  async findWinnerInfoList(context: Context, { session }, [awardsService]: [AwardsService]) {
    const {
      activityId = '',
      prizeId = null,
      userNick = '',
      shipStatus = null,
      prizeType = null,
      prizeDataType = null,
      pageNo = 1,
      pageSize = 10,
      startTime,
      endTime
    } = context.data

    // 获取活动中奖列表
    let awardList = await awardsService.getAwardListByActivityId({
      activityId,
      prizeId,
      userNick,
      shipStatus,
      type: prizeType,
      prizeDataType,
      pageNo,
      pageSize,
      startTime,
      endTime
    })

    console.log(`awardList: ${JSON.stringify(awardList)}`)
    return resultsModel.success(awardList)
  }

  //获取活动中奖名单 为导出做准备
  @paramsCheck(['activityId', 'title'])
  @services([AwardsService])
  async getAwardsList(context: Context, { session }, [awardsService]: [AwardsService]) {
    let { cloud } = context
    let {
      activityId,
      title,
      prizeId,
      userNick,
      startTime,
      endTime,
      shipStatus,
      prizeType,
      prizeDataType,
      pageNo,
      pageSize
    } = context.data
    // 获取活动中奖列表
    let awardList = await awardsService.getAwardListByActivityId({
      activityId,
      title,
      prizeId,
      userNick,
      startTime,
      endTime,
      shipStatus,
      type: prizeType,
      prizeDataType,
      pageNo,
      pageSize,
      isExport: true
    })
    return resultsModel.success(awardList)
  }

  /**
   *  导出活动中奖名单
   */
  @paramsCheck(['title'])
  @services([AwardsService])
  async exportAwardsList(context: Context, { session }, [awardsService]: [AwardsService]) {
    let { cloud } = context
    const { title } = context.data

    let res = []
    let { success, data, message } = await this.getAwardsList(context, { session }, [awardsService])
    const { list: awwardList = [], total } = data || {}

    res = [...res, ...awwardList]
    while (total > res.length) {
      context.data.pageNo++
      let { success, data, message } = await this.getAwardsList(context, { session }, [awardsService])
      const { list: awwardList = [] } = data || {}
      if (success) {
        res = [...res, ...awwardList]
      } else {
      }
    }
    let xlsxData = [
      [
        '奖品编号',
        '用户id',
        '中奖时间',
        '地址填写时间',
        '奖品类型',
        '发奖来源',
        '奖品名称',
        '收货人',
        '联系方式',
        '收货地址'
        // '发货状态',
        // '物流公司',
        // '物流单号'
      ]
    ]
    res.forEach((v, index, arr) => {
      let prizeDataType = '/'
      let prizeType = '/'
      let shipTxt = '/'
      if (v) {
        if (v.prizeDataType == 8) prizeDataType = '累计抽奖'
        if (v.prizeDataType == 7) prizeDataType = '阶梯抽奖'
        if (v.prizeDataType == 9) prizeDataType = '邀请助力抽奖'

        if (v.type == PRIZE_TYPE.EQUITY) prizeType = '优惠券'
        if (v.type == PRIZE_TYPE.CREDITS) prizeType = '积分'
        if (v.type == PRIZE_TYPE.OBJECT) {
          prizeType = '实物'
          if (+v.shipStatus == SHIP_STATUS.NO_ADDRESS) shipTxt = '未填写地址'
          if (+v.shipStatus == SHIP_STATUS.NO_SHIP) shipTxt = '未发货'
          if (+v.shipStatus == SHIP_STATUS.HAS_SHIP) shipTxt = '已发货'
        }
        if (v.type == PRIZE_TYPE.THANK) prizeType = '谢谢参与'
      }
      xlsxData.push([
        v._id,
        v.userNick,
        v.createTime ? formatTime(new Date(v.createTime + EIGHT_HOURS)) : '',
        v.type == 3 ? (v.receiveTime ? formatTime(new Date(v.receiveTime + EIGHT_HOURS)) : '暂未填写') : '/',
        prizeType,
        prizeDataType,
        v.type == 2 ? `${v.credits || 0}积分` : v.name || '',
        v.type == 3 ? v.receiveName || '' : '/',
        v.type == 3 ? (v.phone ? v.phone + '' : '暂未填写') : '/',
        v.type == 3 ? v.province + v.city + v.area + v.streetName + v.addressDetail || '暂未填写' : '/'
        // shipTxt,
        // v.type == 3 ? v.shipCompany || '' : '/',
        // v.type == 3 ? (v.shipNum ? v.shipNum + '' : '') : '/'
      ])
    })
    let buffer = xlsx.build([
      {
        name: title + new Date().getTime(),
        data: xlsxData
      }
    ])
    console.log(`xlsxData: ${JSON.stringify(xlsxData)}`)

    let result = await cloud.file.uploadFile({
      fileContent: buffer,
      fileName: title + new Date().getTime() + '.xlsx'
    })
    // result.url 需进行处理
    if (result.url) {
      result.url = result.url.replace('http', 'https').replace('-internal', '')
    }
    return resultsModel.success(result)
  }

  /**
   *  导入发货
   */
  @paramsCheck(['activityId', 'fileName'])
  @services([AwardsService])
  async downloadFile(context: Context, { session }, [awardsService]: [AwardsService]) {
    const { cloud } = context
    //获取活动id
    const { activityId, fileName } = context.data
    let total = 0
    let result
    try {
      result = await cloud.file.downloadFile({ fileId: fileName })
    } catch {
      return resultsModel.error(CODE_TYPES.SYSTEM_ERROR, 'fileId错误')
    }

    const sheets = xlsx.parse(result)
    console.log(sheets, 'sheets-------')
    //接口调用
    let { data } = sheets[0]
    console.log(JSON.stringify(data), 'data----------')
    if (!data[1] || data[1].length !== 0) {
      if (data[1].length <= 3) {
        for (let i = 1; i < data.length; i++) {
          let prizeId = data[i][0]
          let shipCompany = data[i][1]
          let shipNum = data[i][2]
          // 此处需要判断用户是否填写了地址

          let result2 = await awardsService.setUserShipStatus(activityId, prizeId, shipCompany, shipNum)
          total += result2
        }
        console.log(total, data, 'total,data')
        return resultsModel.success({ total, data })
      } else {
        console.log(data[1].length, 'length')
        return resultsModel.error(CODE_TYPES.SYSTEM_ERROR, '导入文件格式错误')
      }
    } else {
      return resultsModel.error(CODE_TYPES.SYSTEM_ERROR, '导入文件出现错误')
    }
  }
}
