/** @format */

import { GIFT_CARD_STATUS, JOIN_NEED_TIMES, PRIZE_DATA_TYPE, TASK_OPEN_STATUS } from '../constants'
import { services, checkParams, preCheck, preUpdate } from '../decorator/common'
import { CODE_TYPES } from '../errorCode'
import { resultsModel } from '../sdk'
import { CommonCardService, CommonAwardsService } from '../service/common'
import { getToday } from '../utils'
import { checkActivityTime, checkUserInfo, checkVip, checkInviteId, checkInviteUserCard } from '../utils/common/check'
import checkCardGameTimes from '../utils/common/check/checkCardGameTimes'
import { reduceCardGameTimes, updateUserInfo } from '../utils/common/update'

export interface ISumitGameControllerInfos extends IControllerInfos {
  joinInfo: IJoinRecord
}

export default class Card {
  @checkParams(['activityId'])
  @services([CommonCardService, CommonAwardsService])
  async getCollectCardInfo(
    context: IContext<IParams>,
    { userInfo }: IControllerInfos,
    [cardService]: [CommonCardService, CommonAwardsService]
  ) {
    const { activityId } = context.data
    const { joinedTimes, gameTimes, gameTimesEverday } = userInfo

    const myCardInfo = cardService.getMyCardInfo(userInfo)

    const joinedTimesPrizeList = await cardService.getCardPrizeList(
      activityId,
      PRIZE_DATA_TYPE.JOIN_TIMES,
      'needTimes',
      joinedTimes
    )

    const collectedCardTypePrizeList = await cardService.getCardPrizeList(
      activityId,
      PRIZE_DATA_TYPE.CARD_TYPE_AWARD,
      'needCards',
      myCardInfo.cardTypeCollectedCount
    )

    const today = getToday()

    return resultsModel.success({
      joinedTimes,
      gameTimes: gameTimes + (gameTimesEverday?.[today] || 0),
      giftCardStatus: GIFT_CARD_STATUS,
      taskOpenStatus: TASK_OPEN_STATUS,
      ...myCardInfo,
      joinedTimesPrizeList,
      collectedCardTypePrizeList
    })
  }

  @checkParams(['activityId'])
  @services([CommonCardService])
  @preCheck([checkActivityTime, checkVip, checkCardGameTimes])
  @preUpdate([reduceCardGameTimes(JOIN_NEED_TIMES)])
  async doJoin(context: IContext<IParams>, { userInfo }: IControllerInfos, [cardService]: [CommonCardService]) {
    const { activityId } = context.data

    const cardResult = await cardService.collectCard(activityId, userInfo)

    await cardService.addCollectRecord(userInfo, cardResult)

    return resultsModel.success(cardResult)
  }

  // 获取中奖轮播
  @checkParams(['activityId'])
  @services([CommonCardService, CommonAwardsService])
  async getAwardsCarouselList(
    context: IContext<IParams>,
    { userInfo, activityInfo }: IControllerInfos,
    [cardService, awardsService]: [CommonCardService, CommonAwardsService]
  ) {
    const { activityId } = context.data

    const list = await cardService.getAwardsCarouselList(activityId)

    return resultsModel.success({
      list
    })
  }

  // 我的卡牌详情
  @checkParams(['activityId'])
  @services([CommonCardService, CommonAwardsService])
  async getMyCardsInfo(context: IContext<IParams>, { userInfo }: IControllerInfos, [cardService]: [CommonCardService]) {
    const { activityId } = context.data
    const list = await cardService.getMyCardsInfo(activityId, userInfo)

    return resultsModel.success({
      list
    })
  }

  // 获取好友赠送卡片
  @checkParams(['activityId', 'type', 'inviteId'])
  @services([CommonCardService, CommonAwardsService])
  @preCheck([checkActivityTime, checkInviteId, checkInviteUserCard])
  async getGiftCard(
    context: IContext<IParams>,
    { userInfo }: IControllerInfos,
    [cardService]: [CommonCardService],
    { inviteUserInfo }: IPreCheckData
  ) {
    const { activityId, type } = context.data

    const giftResult = await cardService.getGiftCard(activityId, type, userInfo, inviteUserInfo)

    return resultsModel.success(giftResult)
  }
}
