/** @format */

import { services, checkParams, registeInfos } from '../decorator/common'
import { resultsModel } from '../sdk'
import { CommonAwardsService } from '../service/common'
import preCheck from '../decorator/common/preCheck'
import {
  checkActivityTime,
  checkPrizeExpired,
  checkValidEnamePrize,
  checkValidObjectPrize
} from '../utils/common/check/'

export interface IReceivePrizeControllerInfos extends IControllerInfos {
  awardInfo: IAwards
}

export default class AwardsController {
  /**
   *  我的奖品列表
   */
  @checkParams(['activityId'])
  @services([CommonAwardsService])
  async getMyPrizeList(
    context: IContext<IParams>,
    { activityInfo }: IControllerInfos,
    [awardSer]: [CommonAwardsService]
  ): Promise<IResult<{ list: IAwards[] }>> {
    const result = await awardSer.getMyPrizeList(activityInfo)
    return resultsModel.success(result)
  }

  /**
   *  领取实物
   */
  @checkParams(['activityId', 'id', 'name', 'phone', 'province', 'city', 'area', 'addressDetail', 'streetName'])
  @services([CommonAwardsService])
  @preCheck([checkPrizeExpired, checkValidObjectPrize])
  async receiveObjectPrize(
    context: IContext<IParams>,
    { activityInfo }: IReceivePrizeControllerInfos,
    [awardSer]: [CommonAwardsService]
  ) {
    let result = await awardSer.recieveObjectPrize(context)

    return resultsModel.success(result)
  }

  /**
   *  权益重新领取
   */
  @checkParams(['activityId', 'id'])
  @registeInfos(['session'])
  @services([CommonAwardsService])
  @preCheck([checkPrizeExpired, checkValidEnamePrize])
  async receiveEnamePrize(
    context: IContext<IParams>,
    { session }: IReceivePrizeControllerInfos,
    [awardSer]: [CommonAwardsService],
    { awardInfo }: IPreCheckData
  ) {
    let { id } = context.data

    let result = await awardSer.recieveEnamePrize(id, awardInfo, session)

    return resultsModel.success(result)
  }

  /**
   *  概率抽奖
   */
  @checkParams(['activityId', 'prizeDataType'])
  @services([CommonAwardsService])
  @preCheck([checkActivityTime])
  async drawLotteryPrize(
    context: IContext<IParams>,
    { userInfo }: IControllerInfos,
    [awardSer]: [CommonAwardsService]
  ): Promise<
    IResult<{
      id?: string
      type: number
      name: string
      image?: string
    }>
  > {
    const { activityId, prizeDataType } = context.data

    const drawResult = await awardSer.drawLottery(activityId, prizeDataType, userInfo)

    return resultsModel.success(drawResult)
  }
}
