const BaseDao = require("../dao/base.dao");
const {
  ACTIVITY_BASE_CONFIG,
  ACTIVITY_PRIZE_CONFIG,
} = require("../config/db_config");
const { DELETE_STATUS, OPEN_PRIZE_STATUS } = require("../utils/constants");
module.exports = class ActivityConfigService {
  constructor(context) {
    this.activityconfigDao = new BaseDao(context, ACTIVITY_BASE_CONFIG);
    this.activityconfigPrizeDao = new BaseDao(context, ACTIVITY_PRIZE_CONFIG);
  }

  //保存新建活动
  async saveNewActivity(data, prize, openId) {
    try {
      let result = await this.activityconfigDao.insertOne({
        ...data,
        createTime: Date.now(),
        updateTime: Date.now(),
        deleteStatus: DELETE_STATUS.EXIST,
        openPrizeStatus: OPEN_PRIZE_STATUS.WAIT_AWARD,
      });
      console.log(result, "保存基础配置result");
      if (result) {
        try {
          prize.map((item) => {
            item.activityId = result;
            item.openId = openId;
            item.createTime = Date.now();
            item.updateTime = Date.now();
            item.deleteStatus = DELETE_STATUS.EXIST;
            return item;
          });
          return await this.activityconfigPrizeDao.insertMany(prize);
        } catch (e) {
          console.log(e, "保存奖品配置出错");
          // 如果奖品未保存删除之前的活动
          await this.activityconfigDao.deleteMany({
            _id: result,
          });
        }
      }
    } catch (e) {
      //保存活动失败
      console.log(e, "保存活动失败");
    }
  }

  // 更新活动基本信息
  async updateBaseConfigInfo(id, data) {
    return await this.activityconfigDao.update(
      {
        _id: id,
      },
      {
        $set: {
          ...data,
          updateTime: Date.now(),
          deleteStatus: DELETE_STATUS.EXIST,
        },
      }
    );
  }

  // 通过id获取奖品信息
  async getPrizeInfoById(id) {
    let result = await this.activityconfigPrizeDao.find({
      _id: id,
    });
    if (result && result[0]) {
      return result[0];
    }
    return false;
  }

  // 奖品置为删除状态
  async updateDeleteStatusById(id) {
    return await this.activityconfigPrizeDao.update(
      {
        _id: id,
      },
      {
        $set: {
          deleteStatus: DELETE_STATUS.DELETE,
        },
      }
    );
  }

  // 更新单条数据
  async updateOnePrizeById(id, data) {
    return await this.activityconfigPrizeDao.update(
      {
        _id: id,
      },
      {
        $set: {
          ...data,
          updateTime: Date.now(),
          deleteStatus: DELETE_STATUS.EXIST,
        },
      }
    );
  }

  // 插入单条奖品数据
  async insertPrizeByActivityId(activityId, data) {
    return await this.activityconfigPrizeDao.insertOne({
      ...data,
      activityId,
      createTime: Date.now(),
      updateTime: Date.now(),
      deleteStatus: DELETE_STATUS.EXIST,
    });
  }

  // 获取奖品所有配置
  async getAllOldPrizeList(activityId) {
    return await this.activityconfigPrizeDao.find({
      activityId,
    });
  }

  //更新活动配置
  async updateActivity(activityId, data, prizeNoId) {
    console.log(
      activityId,
      data,
      prizeNoId,
      "activityId, data, prizeNoId, oldPrizeIds"
    );
    try {
      let originalData = await this.activityconfigDao.findOne({
        _id: activityId,
      });
      let result = await this.activityconfigDao.update(
        {
          _id: activityId,
        },
        {
          $set: {
            ...data,
            createTime: originalData.createTime,
            updateTime: Date.now(),
          },
        }
      );
      if (result) {
        try {
          let prizeOldIdsByActivity = (
            await this.activityconfigPrizeDao.find({
              activityId,
            })
          ).reduce((s, v) => {
            return (s = [...s, v._id]);
          }, []);
          let prizeResult = await this.activityconfigPrizeDao.insertMany(
            prizeNoId
          );
          if (prizeResult) {
            //删除之前的奖品
            let deleteOldPrizeList = prizeOldIdsByActivity.reduce((s, item) => {
              return (s = [
                ...s,
                this.activityconfigPrizeDao.deleteMany({
                  _id: item,
                }),
              ]);
            }, []);
            return await Promise.all(deleteOldPrizeList);
          }
        } catch (e) {
          await this.activityconfigDao.update(
            {
              _id: activityId,
            },
            {
              $set: {
                ...originalData[0],
              },
            }
          );
        }
      }
    } catch (e) {
      //更新失败
    }
  }

  //通过openId去获取当前商家活动列表
  async getActivityListByOpenId(openId, { pageSize = 10, pageNo = 1 }) {
    return {
      list: await this.activityconfigDao.find(
        {
          openId,
          deleteStatus: DELETE_STATUS.EXIST,
        },
        {
          projection: {
            startTime: 1,
            endTime: 1,
            title: 1,
          },
          sort: {
            createTime: -1,
          },
          limit: pageSize,
          skip: (pageNo - 1) * pageSize,
        }
      ),
      total: await this.activityconfigDao.count({
        openId,
        deleteStatus: DELETE_STATUS.EXIST,
      }),
      pageSize,
      pageNo,
    };
  }


  //通过activityId去获取活动配置信息
  async getActivityInfoByActivityId(activityId) {
    return {
      baseConfig: await this.activityconfigDao.findOne(
        {
          _id: activityId,
          deleteStatus: DELETE_STATUS.EXIST,
        },
        {
          projection: {
            commandImg: 1,
            rule: 1,
            taskList: 1,
            title: 1,
            logoImg: 1,
            prizeInfoList: 1,
            subtitle: 1,
            commandTitle: 1,
            startTime: 1,
            endTime: 1,
            beenInvitedText: 1,
          },
        }
      ),
      prizeConfig: await this.activityconfigPrizeDao.find(
        {
          activityId,
          deleteStatus: DELETE_STATUS.EXIST,
        },
        {
          sort: {
            level: 1,
          },
        }
      ),
    };
  }

  //通过activityId去删除活动
  async deleteActivityByActivityId(activityId) {
    try {
      let result = await this.activityconfigDao.update(
        {
          _id: activityId,
        },
        {
          $set: {
            deleteStatus: DELETE_STATUS.DELETE,
          },
        }
      );
      if (result) {
        return await this.activityconfigPrizeDao.update(
          {
            activityId,
          },
          {
            $set: {
              deleteStatus: DELETE_STATUS.DELETE,
            },
          }
        );
      }
    } catch (e) {
      console.log(e, "删除活动失败");
    }
  }

  // 查询待开奖列表
  async getNodifyList(serverTime) {
    let list = await this.activityconfigDao.find({
      endTime: {
        $lt: serverTime,
      },
      openPrizeStatus: {
        $in: [OPEN_PRIZE_STATUS.WAIT_AWARD, OPEN_PRIZE_STATUS.FAIL],
      },
      deleteStatus: DELETE_STATUS.EXIST,
    });
    console.log(list);
    return list;
  }

  // 更新openPrizeStatus为处理中
  async update2Process(_id) {
    try {
      await this.activityconfigDao.update(
        {
          _id,
        },
        {
          $set: {
            openPrizeStatus: OPEN_PRIZE_STATUS.PROCESSING,
            updateTime: Date.now(),
          },
        }
      );
      return true;
    } catch (e) {
      // 日志记录
      console.log(e);
      return false;
    }
  }

  // 更新openPrizeStatus为成功
  async update2Success(_id) {
    try {
      await this.activityconfigDao.update(
        {
          _id,
        },
        {
          $set: {
            openPrizeStatus: OPEN_PRIZE_STATUS.SUCCESS,
            openPrizeMsg: "",
            updateTime: Date.now(),
          },
        }
      );
      return true;
    } catch (e) {
      // 日志记录
      console.log(e);
      return false;
    }
  }

  // 更新openPrizeStatus为失败
  async update2Fail(_id, message) {
    try {
      await this.activityconfigDao.update(
        {
          _id,
        },
        {
          $set: {
            openPrizeStatus: OPEN_PRIZE_STATUS.FAIL,
            openPrizeMsg: message || "",
            updateTime: Date.now(),
          },
        }
      );
      return true;
    } catch (e) {
      // 日志记录
      console.log(e);
      return false;
    }
  }
};
