import { passUrlList } from "./helper";

// 正整数
export const validatePositiveInteger = number => /^[1-9]\d*$/.test(number);

// 整数
export const validateInteger = number => /^-?\d*$/.test(number);

// 判断是否为数字
export const isNumber = val => {

  var regPos = /^\d+(\.\d+)?$/; // 非负浮点数
  var regNeg = /^(-(([0-9]+\.[0-9]*[1-9][0-9]*)|([0-9]*[1-9][0-9]*\.[0-9]+)|([0-9]*[1-9][0-9]*)))$/; // 负浮点数
  if (regPos.test(val) && regNeg.test(val)) {
    return true;
  } else {
    return false;
  }

};

/**
*
* 验证一定范围内数,支持整数或小数
* @param {*} number 
* @param {*} [ min, max ] 范围    [0,99]  [-Infinity, 0]  [999, Infinity]
* @param {number} [fixed=0] 小数位数  0时为整数
* @returns { boolean }
*/
export const validateRangeNumber = (number, [min, max] = [], fixed = 0) => {
  let pattern = null
  // 判断空字符
  if (typeof number === 'string' && !number.trim()) return false;

  if (fixed === 0 || !number.includes('.')) {
    if (min > 0) {
      // 校验正整数
      pattern = /^[1-9]\d*$/;
    } else if (min === 0 ) {
      // 校验自然数
      pattern = /^([1-9]\d*|[0]{1,1})$/;
    }
  } else {
    pattern = new RegExp(`^[0-9]+(\\.[0-9]{1,${fixed}})?$`)
  }

  if (min > max) {
    console.error('数字范围的最小值不能大于最大值');
    return;
  }

  if (!pattern.test(number)) return false;

  if (number < min || number > max) return false;

  return true;
};

// 单个奖池总概率验证
export const rateTableValidator = (value) => {
  let n = 0;
  let message = ''
  if (!value.length || !value) {
    return message = '请至少配置一个奖品'
  }
  if (value.length) {
    value.forEach(item => {
      n += Number(item.probability);
    });
  }
  const backUp = value.filter(item => item.isBackUp);
  if (backUp.length > 1) {
   return  message = '降级奖品最多配置一个'
  }
  if (n > 100) {
    return message = '总概率不能大于100%'
  }
  return message
};

// 排名表格验证
export const rankTableValidator = (value) => {
  let message = ''
  if (!value.length) return message = '请至少配置一个奖品';
  // rank字段不存在或为0
  if (value.some(v => !v.rank)) return message = '名次配置错误, 请检查';
  if (value.some(v => !v.name)) return message = '名次配置错误, 请检查';
  const rankArr = (value.map(v => v.rank.split('-')));

  // 判断是从第1名开始，因为之前就做了排序，所以判断有 1 就可以了
  const rankArrStr = rankArr.toString();
  if (rankArrStr.indexOf('1') === -1) return message = '必须从第一名开始配置';

  // 判断排名连续
  let validInfo = {
    status: true,
    message: '名次索引必须连续, 请检查'
  };

  rankArr.forEach((rank, i) => {
    if (i === 0) return;
    let lastRank = rankArr[i - 1][rankArr[i - 1].length - 1];
    let curRank = rank[0];
    if (curRank - lastRank === 0) {
      validInfo = {
        status: false,
        message: '请勿对重复的名次添加奖品'
      };
    }
    if (validInfo.status && (curRank - lastRank !== 1)) {
      validInfo.status = false;
    }
  });
  if (!validInfo.status) return message = validInfo.message
  return message;
};

// 活动时间验证
export const rankTimeValidator = (rule, value, callback, source, options) => {
  if (value && value.length == 2) {
    const [startTime, endTime] = value;
    if (!startTime || !endTime) {
      return new Error('请选择完整活动时间');
    }
    // 活动未开始 | 新建活动
    if (+source.originalStartTime > Date.now() || !source.originalStartTime) {
      if (new Date().getTime() > startTime) {
        return new Error('活动开始时间需大于当前时间');
      }
      if (endTime === startTime) {
        return new Error('活动开始时间不能等于结束时间');
      }
      if (startTime > endTime) {
        return new Error('活动开始时间不能大于结束时间');
      }
    }
  }
  return callback();
};

// 判断网站域名
export const urlChecks = (rule, value, callback, source, options) => {
  if (!value) return new Error('请配置链接');
  if (!value.startsWith("http://") && !value.startsWith("https://")) return new Error('请配置链接');

  let passUrl = false;
  passUrlList.some((i) => {
    if (~value.indexOf(i)) {
      passUrl = true;
      return true;
    }
  });
  if (!passUrl) {
    return new Error('域名校验不通过');
  }
  return callback();

};

// 判断是否含有小数点
export const isDot = (num) => {
  var result = (num.toString()).indexOf(".");
  if (result != -1) {
    return false;
  } else {
    return true;
  }
};

/**
 * 校验结果
 * @param {boolean} status 校验状态
 * @param {string} message 错误提示
 */
export const valiResultModal = (status, message = '') => {
  return {
    status,
    message: status ? '' : message
  }
}