/** @format */
import { services, checkParams, registeInfos, preCheck, preUpdate } from '../../decorator/common'
import { checkActivityTime, checkTaskLimit, checkExchangeCreditsTask, checkRemainTimes } from '../../utils/common/check'
import { updateVip, updateSignTask, updateOrderGoods } from '../../utils/common/update'
import customUtil from '../../utils/custom'
import { resultsModel } from '../../sdk'
import { CommonTaskService, CommonBaseService } from '../../service/common'

export default class TaskController {
  /**
   *  任务列表
   */
  @checkParams(['activityId'])
  @registeInfos(['vipInfo'])
  @services([CommonTaskService])
  @preCheck([checkActivityTime])
  @preUpdate([updateVip, updateSignTask, updateOrderGoods])
  async getTaskList(
    context: IContext<IParams>,
    { activityInfo, userInfo, vipInfo }: IControllerInfos,
    [commonTaskService]: [CommonTaskService]
  ) {
    const commonTaskInitTaskListResult1 = await commonTaskService.initTaskList(userInfo, activityInfo)
    return resultsModel.success({
      ...commonTaskInitTaskListResult1
    })
  }
  /**
   *  完成任务
   */
  @checkParams(['activityId', 'taskType'])
  @registeInfos(['session'])
  @services([CommonTaskService])
  @preCheck([checkActivityTime, checkTaskLimit, checkExchangeCreditsTask])
  async doCompleteTask(
    context: IContext<IParams>,
    { activityInfo, userInfo, session }: IControllerInfos,
    [commonTaskService]: [CommonTaskService]
  ) {
    const commonTaskCompleteTaskResult1 = await commonTaskService.completeTask(
      context.data.taskType,
      activityInfo,
      userInfo
    )
    return resultsModel.success({
      ...commonTaskCompleteTaskResult1
    })
  }
  /**
   *  领取任务奖励
   */
  @checkParams(['activityId', 'taskType', 'rewardsKey'])
  @services([CommonTaskService])
  @preCheck([checkActivityTime, checkRemainTimes])
  async receiveTaskRewards(
    context: IContext<IParams>,
    { activityInfo, userInfo }: IControllerInfos,
    [commonTaskService]: [CommonTaskService]
  ) {
    const commonTaskReceiveTaskRewardsResult1 = await commonTaskService.receiveTaskRewards(
      context.data.taskType,
      context.data.rewardsKey,
      userInfo
    )
    return resultsModel.success({
      ...commonTaskReceiveTaskRewardsResult1
    })
  }
  /**
   *  获取收藏商品列表
   */
  @checkParams(['activityId', 'itemIds'])
  @registeInfos(['session'])
  @services([CommonBaseService, CommonTaskService])
  async getCollectGoodsList(
    context: IContext<IParams>,
    { activityInfo, userInfo, session }: IControllerInfos,
    [commonBaseService, commonTaskService]: [CommonBaseService, CommonTaskService]
  ) {
    const commonBaseGetItemListByIdsResult1 = await commonBaseService.getItemListByIds(context.data.itemIds, session)
    const commonTaskGetItemListWithCollectStatusResult2 = await commonTaskService.getItemListWithCollectStatus(
      commonBaseGetItemListByIdsResult1,
      userInfo
    )
    return resultsModel.success({
      ...commonTaskGetItemListWithCollectStatusResult2
    })
  }
}
