import {useState, useEffect, useRef} from 'react'
import API from '@/api'
import {goodsListInfoModel} from '@/store'
import { useThrottle } from './useThrottle';

const pageSize = 8;

export const useGoods = (props) => {

    const {key, itemIds} = props;

    const itemIdsArray = useRef(itemIds.split(','));

    // 当前索引值
    const itemIndex = useRef(0)

    const {listStorage, setListStorage} = goodsListInfoModel();

    const [goodsList, setGoodsList] = useState([]);

    useEffect(()=>{
        // 小于分页 请求商品数
        if(itemIdsArray.current.length < pageSize){
            getGoodsList(itemIds)
            return
        }
        // 存储当前标识 
        itemIndex.current = pageSize;
        // 多于分页请求商品数量
        getGoodsList(itemIdsArray.current.slice(0, pageSize).join(','))
    },[])

    // 商品列表请求
    const getGoodsList = async (itemIds,isReset = false) => {
        // 判断是否存在listStorage[key]
        if(listStorage?.[key] && listStorage?.[key]?.length > goodsList.length){
            setGoodsList(listStorage?.[key]);
            return;
        }
        const { data } = await API.getCollectGoods({ key, itemIds })
        if(data?.list){
            const list = isReset ? [...data?.list] : [...goodsList, ...data?.list];
            // 缓存当前key对应的goodslist
            setListStorage({
                ...listStorage,
                [key]:list
            })
            setGoodsList(list || [])
        }
    }
    // 下一页 商品列表请求
    const scrollToNextList = useThrottle(()=>{

        console.log(itemIdsArray.current.length , pageSize,itemIndex.current, itemIdsArray.current.length);
        // 没有第二页
        if(itemIdsArray.current.length < pageSize || itemIndex.current == itemIdsArray.current.length) return
        let curretnIndex = 0;
        // 第二页是不是最后一页
        if(itemIndex.current + pageSize > itemIdsArray.current.length) {
            curretnIndex = itemIdsArray.current.length;
        } else {
            // 第二页满数量
            curretnIndex = itemIndex.current + pageSize;
        }
        // 请求接口
        getGoodsList(itemIdsArray.current.slice(itemIndex.current, curretnIndex).join(','));
        // 索引值更新
        itemIndex.current = curretnIndex;
    })

    const updateGoodsList = async () => {
        if(itemIdsArray.current.length <= pageSize) {
            await getGoodsList(itemIds,true)
            return;
        }
        // 当前页面的索引值
        const preInx = (Math.ceil(itemIndex.current / pageSize) - 1) * pageSize
        // 当前页面的所有itemId
        const _itemIds = itemIdStr.current.slice(preInx, itemIdInx.current).join(',')
        // 更新当前商品的状态
        const goodRes = await getCollectGoods({ key, itemIds: _itemIds })
        const newGoodsList = [...(goodList.slice(0,preInx)),...(goodRes?.data?.list || [])]

        setListStorage({
            ...listStorage,
            [key]:newGoodsList
        })
        
        setGoodList(newGoodsList)
    }

     // 完成任务
    const completeTask = useThrottle( async (callbackFn=()=>{}) => {
        const res = await API.completeTask({ key, itemId})
        if(res?.success){
            updateGoodsList();
            callbackFn && callbackFn();
        }
    })

    return {
        goodsList,
        scrollToNextList,
        completeTask,
    }
}