/** @format */
import { services, checkParams, registeInfos, preCheck, preUpdate } from '../../decorator/common'
import {
  checkActivityTime,
  checkGrowTaskLimit,
  checkExchangeCreditsTask,
  checkRemainTimes
} from '../../utils/common/check'
import { updateVip, updateSignGrowTask, updateGrowOrderGoods } from '../../utils/common/update'
import customUtil from '../../utils/custom'
import { resultsModel } from '../../sdk'
import { CommonGrowtaskService, CommonBaseService } from '../../service/common'

export default class TaskController {
  /**
   *  任务列表
   */
  @checkParams(['activityId'])
  @registeInfos(['vipInfo', 'task'])
  @services([CommonGrowtaskService])
  @preCheck([checkActivityTime])
  @preUpdate([updateVip, updateSignGrowTask, updateGrowOrderGoods])
  async getTaskList(
    context: IContext<IParams>,
    { activityInfo, userInfo, vipInfo, task }: IControllerInfos,
    [commonGrowtaskService]: [CommonGrowtaskService]
  ) {
    const commonGrowtaskInitTaskListResult1 = await commonGrowtaskService.initTaskList(userInfo, activityInfo, task)
    return resultsModel.success({
      ...commonGrowtaskInitTaskListResult1
    })
  }
  /**
   *  完成任务
   */
  @checkParams(['activityId', 'taskType'])
  @registeInfos(['session', 'task'])
  @services([CommonGrowtaskService])
  @preCheck([checkActivityTime, checkGrowTaskLimit, checkExchangeCreditsTask])
  async doCompleteTask(
    context: IContext<IParams>,
    { activityInfo, userInfo, session, task }: IControllerInfos,
    [commonGrowtaskService]: [CommonGrowtaskService]
  ) {
    const commonGrowtaskCompleteTaskResult1 = await commonGrowtaskService.completeTask(
      context.data.taskType,
      activityInfo,
      userInfo,
      task
    )
    return resultsModel.success({
      ...commonGrowtaskCompleteTaskResult1
    })
  }
  /**
   *  领取任务奖励
   */
  @checkParams(['activityId', 'taskType', 'rewardsKey'])
  @services([CommonGrowtaskService])
  @preCheck([checkActivityTime, checkRemainTimes])
  async receiveTaskRewards(
    context: IContext<IParams>,
    { activityInfo, userInfo }: IControllerInfos,
    [commonGrowtaskService]: [CommonGrowtaskService]
  ) {
    const commonGrowtaskReceiveTaskRewardsResult1 = await commonGrowtaskService.receiveTaskRewards(
      context.data.taskType,
      context.data.rewardsKey,
      userInfo
    )
    return resultsModel.success({
      ...commonGrowtaskReceiveTaskRewardsResult1
    })
  }
  /**
   *  获取收藏商品列表
   */
  @checkParams(['activityId', 'itemIds'])
  @registeInfos(['session'])
  @services([CommonBaseService, CommonGrowtaskService])
  async getCollectGoodsList(
    context: IContext<IParams>,
    { activityInfo, userInfo, session }: IControllerInfos,
    [commonBaseService, commonGrowtaskService]: [CommonBaseService, CommonGrowtaskService]
  ) {
    const commonBaseGetItemListByIdsResult1 = await commonBaseService.getItemListByIds(context.data.itemIds, session)
    const commonGrowtaskGetItemListWithCollectStatusResult2 = await commonGrowtaskService.getItemListWithCollectStatus(
      commonBaseGetItemListByIdsResult1,
      userInfo
    )
    return resultsModel.success({
      ...commonGrowtaskGetItemListWithCollectStatusResult2
    })
  }
}
