/**
 * 用户相关方法
 *
 * @format
 */

import { BaseDao } from '../../sdk'
import BaseService from './base.service'
import { USER_DB_NAME } from '../../db'
import { initCustomUserWhenLogin, updateUserWhenLogin } from '../../customUser'
import { getToday, formatUpdateUserProjection } from '../../utils'
import { isNewVip } from '../../utils/common/userUpdate'
import { formatUserNick, formatUpdatedDataByProjection } from '../../utils/common/format'
import { setNewFollowUserData, setNewVipUserData } from '../../utils/common/userUpdate'
import { getShopVip, formatVipCbUrl } from '../../utils/common/vip'
import { get } from 'mpath'
import { merge } from 'lodash'

class UserService extends BaseService {
  userdao: IBaseDao
  constructor(context: IContext<IParams>) {
    super(context)
    this.userdao = new BaseDao(context, USER_DB_NAME)
  }
  /**
   * @desc 获取当前打开活动的用户详情
   * @returns 若用户不存在，返回null; 用户存在，返回用户信息(object对象)
   */
  async getUserInfo(projection = {}) {
    let { openId, data } = this.context
    let { activityId } = data
    return await this.userdao.findOne<IUserInfo>({ openId, activityId }, { projection })
  }

  /**
   * @desc 根据openId获取用户详情
   * @desc 常用于助力分享码为用户openId, 被邀请人打开活动助力时需要获取邀请人的用户详情
   * @param {openId} openId
   * @returns 若用户不存在，返回null; 用户存在，返回用户信息(object对象)
   */
  async getUserInfoByOpenId(openId: string) {
    let { activityId } = this.context.data
    let record = await this.userdao.findOne<IUserInfo>({ openId, activityId })
    return record
  }

  async doLogin(userInfo: IUserInfo, vipInfo: IVipInfo, activityInfo: IActivityInfo) {
    if (!userInfo) {
      const customUserInfo = initCustomUserWhenLogin(this.context, activityInfo)
      userInfo = await this.initUserData(vipInfo, activityInfo, customUserInfo)
    } else {
      userInfo = await this.updateUserData(vipInfo, userInfo, activityInfo)
    }
    return userInfo
  }

  /**
   * 初始化用户数据
   *
   * @param {IVipInfo} vipInfo
   * @param {IActivityInfo} activityInfo
   * @returns
   * @memberof UserService
   */
  async initUserData(vipInfo: IVipInfo, activityInfo: IActivityInfo, customUserInfo: ICustomUserInfo) {
    const { openId, data } = this.context
    const { activityId, isFollow, avatar, userNick } = data
    const today = getToday()
    console.log(`----------${today}----------`, today)
    const { tasks } = activityInfo
    const { follow = {} as ITaskConfig, member = {} as ITaskConfig } = tasks
    const user: ICommonUserInfo = {
      activityId,
      avatar,
      userNick,
      openId,
      remainTimes: {
        follow: isFollow ? follow.value : 0,
        member: vipInfo.isVip ? member.value : 0,
        ...initRemainTimes(tasks)
      },
      member: {
        flag: !!vipInfo.isVip
      },
      follow: {
        flag: !!isFollow
      },
      login: {
        [today]: 1
      },
      taskInfo: {
        [today]: {}
      },
      createTime: Date.now(),
      createDay: today,
      updateTime: Date.now()
    }

    function initRemainTimes(tasks: ITasks) {
      let remainTimes = {} as IRemainTimesInfo
      Object.keys(tasks).forEach(task => {
        if (task !== 'follow' && task !== 'member') {
          remainTimes[task] = 0
        }
      })
      return remainTimes
    }

    // 初始化 如果已关注添加任务记录
    if (isFollow) {
      user.taskInfo[today].follow = [{ createTime: Date.now() }]
    }
    // 初始化 如果已经是会员添加任务记录
    if (vipInfo.isVip) {
      user.taskInfo[today].member = [{ createTime: Date.now() }]
    }

    await this.userdao.insertOne<IUserInfo>({
      ...user,
      ...customUserInfo
    })

    return {
      ...user,
      ...customUserInfo
    }
  }

  async updateUserData(vipInfo: IVipInfo, userInfo: IUserInfo, activityInfo: IActivityInfo): Promise<IUserInfo> {
    const { data } = this.context
    const { isFollow } = data

    const today = getToday()
    let projection: IUpdateQuery = {
      $set: {
        updateTime: Date.now()
      },
      $push: {},
      $inc: {
        [`login.${today}`]: 1
      }
    }

    const { tasks } = activityInfo
    const { follow, member } = tasks

    // 老用户
    // 之前进入活动未关注，现在进入关注，则视为新关注店铺用户
    const followBefore = userInfo.follow.flag
    const followNow = isFollow
    if (!followBefore && followNow && !userInfo.follow.newFollow) {
      projection.$set.follow = setNewFollowUserData(userInfo.follow)
      projection.$push[`taskInfo.${today}.follow`] = { createTime: Date.now() }
      follow?.value && (projection.$inc[`remainTimes.follow`] = follow.value)
    }

    // 之前进入活动非会员，现在进入会员，则视为新会员用户
    const newVip = isNewVip(userInfo, vipInfo)
    if (newVip) {
      projection.$set.member = setNewVipUserData(userInfo.member)
      projection.$push[`taskInfo.${today}.member`] = { createTime: Date.now() }
      member?.value && (projection.$inc[`remainTimes.member`] = member.value)
    }

    projection = merge({}, projection, updateUserWhenLogin(this.context, activityInfo, userInfo))

    await this.updateUser(userInfo._id, projection)

    return formatUpdatedDataByProjection(userInfo, projection)
  }

  /**
   * @desc 更新用户表
   * @param {用户的主键id} _id
   * @param {更新的对象} document
   * @returns 若更新成功，返回为1； 若更新失败，返回为 0 或系统直接报错
   */
  async updateUser(_id: string, projection: IUpdateQuery) {
    return await this.userdao.update({ _id }, formatUpdateUserProjection(projection))
  }

  // 获取排名列表
  async getRank(sortValueKey = 'totalScore', sortTimeKey = 'updateScoreTime', limit = 200) {
    let { activityId } = this.context.data
    //获取排名
    let list = await this.userdao.find<IUserInfo>(
      {
        activityId: activityId,
        [sortValueKey]: { $gt: 0 }
      },
      {
        projection: { [sortValueKey]: 1, userNick: 1, avatar: 1, _id: 0 },
        sort: { [sortValueKey]: -1, [sortTimeKey]: 1 },
        limit
      }
    )

    return {
      list: list.map((v, i) => {
        return {
          [sortValueKey]: v[sortValueKey],
          avatar: v.avatar,
          userNick: formatUserNick(v.userNick),
          rank: i + 1
        }
      })
    }
  }

  /**
   *  获取用户排名
   *
   * @param {string} sortValueKey 排名字段
   * @param {userInfo} IUserInfo
   * @param {string} sortTimeKey
   * @return {number}  排行
   * @memberof UserService
   */
  async getMyRankInfo(sortValueKey = 'totalScore', sortTimeKey = 'updateScoreTime', userInfo: IUserInfo) {
    let { activityId } = this.context.data
    const { openId } = this.context
    const userValue = userInfo[sortValueKey]
    const { userNick, avatar } = userInfo

    let sameScoreList = await this.userdao.find<IUserInfo>(
      { [sortValueKey]: userValue, activityId },
      {
        sort: { [sortTimeKey]: 1 }
      }
    )
    let rank: number
    let gap = 0

    //说明有多个跟自己同分数的人
    for (let j = 0; j < sameScoreList.length; j++) {
      if (sameScoreList[j].openId === openId) {
        gap = j
      }
    }
    rank = await this.userdao.count({ [sortValueKey]: { $gt: userValue }, activityId })
    rank = rank + 1 + gap
    return {
      rank,
      userNick,
      avatar,
      [sortValueKey]: userValue
    }
  }

  /**
   * @desc 根据用户主键id查找用户详情
   * @param {用户的主键id} _id
   * @returns 若用户不存在，返回null; 用户存在，返回用户信息(object对象)
   */
  async getUserInfoById(_id: string) {
    return await this.userdao.findOne({ _id })
  }

  async getShopVip(context: IContext<IParams>, activityInfo: IActivityInfo) {
    const { callbackUrl } = context.data
    return await getShopVip(context, activityInfo, callbackUrl || formatVipCbUrl(context))
  }

  async queryDataByKey(key: string, userInfo: IUserInfo, customKey?: string) {
    return {
      [customKey || key]: get(key, userInfo)
    }
  }
}

export default UserService
