/**
 * 用户相关方法
 */
const {
  BaseDao,
  TBAPIS
} = require('../sdk');

const BaseService = require('./base.service');
const {
  USER_DB_NAME,
  DATA_DB_NAME
} = require('../db');
const {
  TASK_TYPE_WIN
} = require('../constants')

class UserService extends BaseService {
  constructor(context) {
    super(context);
    this.userdao = new BaseDao(context, USER_DB_NAME);
    this.dataDao = new BaseDao(context, DATA_DB_NAME);
  }
  /**
   * @desc 获取当前打开活动的用户详情
   * @returns 若用户不存在，返回null; 用户存在，返回用户信息(object对象)
   */
  async getUserInfo() {
    let {
      openId,
      data
    } = this.context;
    let {
      activityId
    } = data;
    let record = await this.userdao.findOne({
      openId,
      activityId
    });
    return record;
  }

  /**
   * @desc 根据inviteId获取用户详情
   * @desc 常用于助力分享码为用户openId, 被邀请人打开活动助力时需要获取邀请人的用户详情
   * @param {邀请人的openId} inviteId 
   * @returns 若用户不存在，返回null; 用户存在，返回用户信息(object对象)
   */
  async getUserInfoByOpenId(inviteId) {
    let {
      activityId
    } = this.context.data;
    let record = await this.userdao.findOne({
      openId: inviteId,
      activityId
    });
    return record;
  }

  /**
   * @desc 获取是否是会员
   * @param {调用淘宝接口的session} session 
   * @returns {isvip: boolean(是否是会员), url: string(入会链接) }
   */
  async getShopVip(session, callbackUrl = "") {
    // 会员mock，当传递的参数有vipmock: true，则表示已经是会员
    let {
      vipmock = false
    } = this.context.data;
    if (vipmock) {
      return {
        isvip: true
      }
    }
    let result = {}
    let shopUrl = {};
    try {
      result = await TBAPIS.queryVipinfo(this.context, session);
      shopUrl = await TBAPIS.getShopVipUrl(
        this.context, session, {
          "source": "isvapp",
          "entrance": "duiba"
        },
        callbackUrl
      );
    } catch (e) {
      console.log(e);
    }
    console.log(`result, shopUrl`, result, shopUrl);
    return {
      isvip: !!(result.result && result.result.member_info),
      url: shopUrl.result && shopUrl.result.result
    }
  }

  /**
   * @desc 更新用户表
   * @param {用户的主键id} _id 
   * @param {更新的对象} document 
   * @returns 若更新成功，返回为1； 若更新失败，返回为 0 或系统直接报错
   */
  async updateUser(_id, document) {
    console.log(document);
    return await this.userdao.update({
      _id
    }, {
      $set: {
        ...document,
        updateTime: Date.now()
      }
    });
  }
  /**
   * @desc 根据用户主键id查找用户详情
   * @param {用户的主键id} _id 
   * @returns 若用户不存在，返回null; 用户存在，返回用户信息(object对象)
   */
  async getUserInfoById(_id) {
    return await this.userdao.findOne({
      _id
    });
  }

  async insertInitUser(userInfo) {
    let serverTime = Date.now();
    return this.userdao.insertOne({
      ...userInfo,
      createTime: serverTime,
      updateTime: serverTime,
      createDay: getToday(),
    });
  }

  /**
   * 查询用户积分
   */
  async queryUserCredits(session) {
    try {
      let result = await TBAPIS.queryCredits(this.context, session)
      console.log(result)
      return result.result;
    } catch (e) {
      console.log(`查询用户积分失败: `, e);
      return false;
    }
  }

  /**
   * 更新（加扣）用户积分
   * @param {number} quantity 数量
   * @param {string} optType add 增加 reduce 减少
   * @param {number} changeType 变更类型
   */
  async updateUserCredits(
    quantity,
    session,
    optType = 'reduce',
    changeType = 1
  ) {
    const isAdd = optType === 'add';
    try {
      const options = {
        quantity, // 积分数量, 每次变更的数量
        change_type: changeType, //变更类型：0交易，1：互动活动，2：权益兑换，3：手工调整
        opt_type: isAdd ? '0' : 1, // 操作类型，0：增加，1：扣减
        remark: `参与活动` // 备注
      };
      console.log(`${isAdd ? '加' : '扣'}积分请求参数`, options);
      const result = await TBAPIS.changeCredits(this.context, session, options)
      console.log(
        `${isAdd ? '加' : '扣'}积分返回结果:${JSON.stringify(result)}`
      );
      return result;
    } catch (e) {
      console.log(`更新用户积分失败`, e);
      return false;
    }
  }

  // 添加用户
  async addUserInfo(isvip, initialCredits, activityInfo = {}) {
    const {
      openId,
      data: {
        activityId
      }
    } = this.context;
    let userInfo = await this.userdao.findOne({
      openId,
      activityId
    });

    // 新用户
    if (!userInfo) {
      const record = this.initUserData(isvip, initialCredits, activityInfo);
      const result = await this.userdao.insertOne(record);
      console.log(result, '新用户')
      record.isNewUser = true;
      return record;
    }

    // 老用户
    // 之前进入活动非会员，现在进入会员，则视为新会员用户
    const memberBefore = userInfo.member.flag;
    const memberNow = isvip;
    if (!memberBefore && memberNow) {
      userInfo.member.newMember = true;
      userInfo.member.bememberTime = getToday();

      //新用户埋点
      const dataService = new DataService(context);
      dataService.insertUV({
        activityId,
        openId,
        type: DATA_TYPE.newMemberUv
      });
    };
    //更新用户初始积分
    if (initialCredits !== userInfo.initialCredits) {
      userInfo.initialCredits = initialCredits
    }

    const today = getToday();
    if (!userInfo[`taskInfo.everyday.${today}`]) {
      let everyday = {
        [today]: {
          browse: {
            status: 0,
            rewards: TASK_TYPE_WIN.BROWSE,
            done: false
          },
          exchanage: {
            limitTimes: 2,
            times: 0,
            rewards: TASK_TYPE_WIN.EXCHANAGE_CREDITS,
            done: false
          },
          freeMasonrys: {
            rewards: TASK_TYPE_WIN.DAY,
            done: true,
            status: 0,
          }
        }
      }
      userInfo[`taskInfo.everyday`] = everyday;
      console.log('老用户' + userInfo[`taskInfo.everyday`])
    }
    // 添加登录次数
    let loginTimes = userInfo['info'][today]['login'] ? userInfo['info'][today]['login'] + 1 : 1;
    // userInfo['info'][today] = userInfo['info'][today] || {};
    await this.updateUser(userInfo._id, {
      info: {
        [today]: {
          login: loginTimes
        }
      },
      member: userInfo.member,
      masonrys: userInfo.masonrys,
      initialCredits: userInfo.initialCredits,
      taskInfo: {
        everyday: userInfo[`taskInfo.everyday`]
      },
      updateTime: Date.now()
    })
    console.log(userInfo, '旧用户')
    return userInfo;
  }

  // 初始化用户数据
  initUserData(isvip, initialCredits, activityInfo = {}) {
    const {
      openId,
      data: {
        activityId,
        userNick,
        avatar,
        inviteId,
        isFollow
      }
    } = this.context;

    //关注店铺
    let follow = {
      flag: false
    }
    if (isFollow) {
      follow.flag = true;
      follow.newFollow = false;
      follow.followTime = ''
    }
    // 初始化会员情况
    let member = {
      flag: false
    };

    if (isvip) {
      member.flag = true;
      member.newMember = false;
      member.bememberTime = '';
    }

    const record = {
      openId,
      activityId,
      userNick,
      avatar,
      inviteId,
      member,
      follow,
      totalScore: 0,
      maxScore: 0,
      rankPrizePop: 0,
      initialCredits,
      diamond: 10,
      resurgenceCards: 0,
      totalShoes: 0,
      info: {
        [getToday()]: {
          login: 1
        }
      },
      newUserGuide: 0,
      taskInfo: {
        follow: {
          rewards: TASK_TYPE_WIN.FOLLOW,
          done: follow.flag,
          status: 0
        },
        orderGoods: {
          rewards: TASK_TYPE_WIN.FOLLOW,
          done: false,
          waitReceive: 0
        },
        invites: {
          waitReceive: 0,
          done: false,
          rewards: TASK_TYPE_WIN.INVITES,
          value: TASK_TYPE_WIN.NEN_INVITES,
          waitValue: 0
        },
        everyday: {
          [getToday()]: {
            browse: {
              status: 0,
              rewards: TASK_TYPE_WIN.BROWSE,
              done: false
            },
            exchanage: {
              limitTimes: 2,
              times: 0,
              rewards: TASK_TYPE_WIN.EXCHANAGE_CREDITS,
              done: false
            },
            freeMasonrys: {
              rewards: TASK_TYPE_WIN.DAY,
              done: true,
              status: 0
            }
          }
        }
      },
      createTime: Date.now(),
      updateTime: Date.now(),
      createDay: getToday(),
    }
    return record;
  }

  /**
   * 获取用户排名
   */
  async getUserRank(maxScore) {
    let {
      activityId
    } = this.context.data;
    const {
      openId
    } = this.context;
    let sameScoreList = await this.userdao.find({
      maxScore: maxScore,
      activityId: activityId
    }, {
      sort: {
        maxScoreTime: 1
      }
    });
    let rank = "未上榜";
    if (!maxScore) return {
      rank
    };
    sameScoreList = sameScoreList && sameScoreList.filter(i => i.maxScore > 0)
    let gap = 0;
    //说明有多个跟自己同分数的人
    for (let j = 0; j < sameScoreList.length; j++) {
      if (sameScoreList[j].openId == openId) {
        gap = j;
      }
    }
    rank = await this.userdao.count({
      maxScore: {
        $gt: maxScore
      },
      activityId: activityId
    });
    rank = rank + 1 + gap;
    if (rank > 100) {
      rank = '100+';
    }
    return {
      rank
    };
  }

  async getTotalRankUserCount(activityId) {
    return await this.userdao.count({
      activityId,
      maxScoreTime: {
        $exists: true
      }
    });
  }

  // 排行榜, 前100名
  async getRankList(activityId, pageNum = 1, pageSize = 100) {
    console.log(`activityId`, activityId);
    return await this.userdao.find({
      activityId,
      maxScoreTime: {
        $exists: true
      }
    }, {
      projection: {
        userNick: 1,
        avatar: 1,
        maxScore: 1,
        openId: 1
      },
      sort: {
        maxScore: -1,
        maxScoreTime: 1
      },
      limit: pageSize,
      skip: (pageNum - 1) * pageSize,
    });
  }
}

module.exports = UserService;