const UserService = require('../service/user.service');
const ActivityService = require('../service/base.service');
const UserJoinService = require("../service/userJoin.service");
const AccessService = require("../service/access.service");
const StatService = require("../service/stat.service");
const AwardsService = require('../service/awards.service');
const {
  CODE_TYPES,
  DATA_TYPE,
  INVITE_FRIENDS_LIMIT,
  OPEN_PRIZE_STATUS,
  PRIZE_DATA_TYPE
} = require('../constants');
const {
  logger,
  addErrorLog,
  getToday,
  getSellerSession,
  statInsertUV
} = require('../utils')
const {
  ResultsModel
} = require('../sdk');
let resultModel = new ResultsModel();

/**
 * 登陆
 */
const login = async (context) => {
  logger('login', context.data);
  const {
    userNick,
    activityId,
    avatar,
    inviteId = "",
    isFollow = false
  } = context.data;
  const {
    openId
  } = context;
  if (!activityId) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `activityId必填`);
  }
  if (!userNick || !avatar) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `userNick, avatar 必填`);
  }
  try {
    const activityService = new ActivityService(context);
    const userService = new UserService(context);
    const accessService = new AccessService(context);
    const statService = new StatService(context);
    const UserJoinSer = new UserJoinService(context);
    let activityInfo = await activityService.getBaseInfo(activityId);
    // 活动不存在
    if (!activityInfo) {
      return resultModel.error(CODE_TYPES.ERROR_NO_ACTIVITY);
    }
    let {
      session
    } = await getSellerSession(context, activityId);
    // 是否是会员
    let {
      isvip,
      url
    } = await userService.getShopVip(session);

    let joinCount = await UserJoinSer.getJoinCount(activityId, openId);
    // let isvip = false;
    let today = getToday();
    let userInfo = await userService.getUserInfo();
    //访问pv
    await accessService.addAccess(isvip, DATA_TYPE.accesssPv);
    //访问uv
    await accessService.addAccess(isvip, DATA_TYPE.accesssUv);

    //初始积分
    let initialCredits = await userService.queryUserCredits(session) || 0

    if (userInfo) {
      let {
        _id,
        member,
        diamond,
        resurgenceCards
      } = userInfo;
      if (!member.newMember && !member.flag) {
        await userService.updateUser(_id, {
          inviteId,
          initialCredits,
          member: {
            flag: isvip,
            newMember: isvip ? true : false,
          }
        });
      }
      return resultModel.success({
        inviteId: openId,
        diamond,
        resurgenceCards,
        initialCredits: isvip ? initialCredits : 0,
        newUserGuide: joinCount > 0 ? 1 : 0,
        isVip: isvip,
        vipLink: url,
      });
    }
    //首次进入uv
    await statInsertUV(context, {
      openId,
      activityId,
      type: DATA_TYPE.newAddUv
    });

    let result = await userService.insertInitUser({
      userNick,
      activityId,
      avatar,
      openId,
      inviteId,
      diamond: 0,
      totalScore: 0,
      maxScore: 0,
      rankPrizePop: 0,
      initialCredits,
      resurgenceCards: 0,
      totalShoes: 0,
      newUserGuide: 0,
      member: {
        flag: isvip,
        newMember: false,
      },
      follow: {
        flag: isFollow,
        newFollow: false,
      },
      toolInfo: {
        normal: 1,
        primary: 0,
        medium: 0
      },
      remainTimes: {
        [TASK_TYPE.BROWSE]: 0,
        [TASK_TYPE.BUY_GOODS]: 0,
        [TASK_TYPE.INVITES]: 0,
        [TASK_TYPE.FOLLOW]: isFollow ? 1 : 0,
        [TASK_TYPE.FREE_MASONRYS]: 1,
      },
      recordTask: {
        invites: [],
        orders: [],
        followStatus: isFollow ? 1 : 0
      },
      taskInfo: {},
      info: {
        [today]: {
          login: 1
        }
      },
    });
    if (result) {
      return resultModel.success({
        inviteId: openId,
        diamond: 0,
        resurgenceCards: 0,
        initialCredits: isvip ? initialCredits : 0,
        newUserGuide: joinCount > 0 ? 1 : 0,
        isVip: isvip,
        vipLink: url,
      });
    }
    return resultModel.error(CODE_TYPES.ERROR_LOGIN_INSERT);
  } catch (e) {
    console.log(e, "login接口出错了");
    return resultModel.error(CODE_TYPES.SYSTEM_ERROR, "用户登录失败");
  }
};

/**
 * 判断是否VIP
 */
const getVipInfo = async context => {
  logger('getVipInfo', context.data);

  let {
    activityId
  } = context.data
  if (!activityId) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `activityId必填`)
  }
  const {
    session
  } = await getSellerSession(context, activityId);
  if (!session) {
    await addErrorLog(context, {
      functionName: 'getVipInfo',
      errorMessage: '获取卖家session失败'
    });
    return resultModel.error(CODE_TYPES.SYSTEM_ERROR, '获取卖家session失败');
  }
  const userService = new UserService(context);
  let userInfo = await userService.getUserInfo();
  let {
    _id,
    member
  } = userInfo
  try {
    let vipInfo = await userService.getShopVip(session);
    if (!member.flag && vipInfo.isvip) {
      await userService.updateUser(_id, {
        inviteId,
        initialCredits: isVip ? initialCredits : 0,
        member: {
          flag: vipInfo.isvip,
          newMember: true,
          bememberTime: Date.now()
        }
      });
    }
    return resultModel.success(vipInfo)
  } catch (error) {
    console.log('判断是否vip失败' + error)
  }
}


const getRankList = async (context) => {
  logger('getRankList', context.data);
  let {
    activityId
  } = context.data
  if (!activityId) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `activityId必填`)
  }
  const userService = new UserService(context);
  const awardsSer = new AwardsService(context);
  try {
    let {
      maxScore,
      userNick,
    } = await userService.getUserInfo();
    let {
      rank
    } = await userService.getUserRank(maxScore);
    let list = await userService.getRankList(activityId);
    let prizeList = await awardsSer.getPrizeByType(activityId, PRIZE_DATA_TYPE.RANK)
    return resultModel.success({
      detail: {
        rank,
        maxScore,
        userNick
      },
      list: list.map(i => {
        return {
          ...i,
          userNick: i.userNick[0] + "***" + i.userNick[i.userNick.length - 1]
        }
      }),
      prizeList
    })
  } catch (error) {
    console.log('获取排行失败' + error)
  }
}

const rankPrizePop = async (context) => {
  logger('rankPrizePop', context.data);
  let {
    activityId
  } = context.data
  let {
    openId
  } = context;
  if (!activityId) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `activityId必填`)
  }
  const userSer = new UserService(context);
  const awardsSer = new AwardsService(context);
  const userJoinSer = new UserJoinService(context);
  const activitySer = new ActivityService(context);
  try {
    let {
      maxScore,
      rankPrizePop,
      _id: userId
    } = await userSer.getUserInfo();
    let {
      openPrizeStatus
    } = await activitySer.getBaseInfo(activityId);
    let {
      rank
    } = await userSer.getUserRank(maxScore);
    let prize = await awardsSer.getUserAwardPrizeByType(activityId, PRIZE_DATA_TYPE.RANK) || '';
    let userJoinCount = await userJoinSer.getJoinCount(activityId, openId)
    if (prize || openPrizeStatus === OPEN_PRIZE_STATUS.SUCCESS) {
      await userSer.updateUser(userId, {
        rankPrizePop: 1
      })
    }
    return resultModel.success({
      rank,
      prize,
      needPop: rankPrizePop === 0 && userJoinCount > 0
    })
  } catch (error) {
    console.log('获取排行失败' + error)
    return resultModel.error(CODE_TYPES.SYSTEM_ERROR, '获取排行失败')
  }
}

module.exports = {
  // login,
  // getVipInfo,
  getRankList,
  rankPrizePop
}