const {
  pipeActivityInfo
} = require('./base.controller');
const UserService = require('../service/user.service');
const AwardsService = require('../service/awards.service');
const {
  CODE_TYPES,
  PRIZE_DATA_TYPE,
  DRAW_STATUS,
  PRIZE_TYPE,
  SHIP_TYPE,
  HAS_EXCHANGE
} = require('../constants');
const {
  ResultsModel
} = require('../sdk');
let resultModel = new ResultsModel();

/**
 * 我的奖品列表
 * @param {*} context 
 */
const getMyPrizeList = async (context) => {
  // 判断活动
  let pipeRes = await pipeActivityInfo(context);
  // 活动结束可以查看我的奖品
  if (pipeRes.code && pipeRes.code !== CODE_TYPES.ERROR_ACTIVITY_OVER.code) {
    return pipeRes;
  }
  let awardSer = new AwardsService(context);
  let result = await awardSer.getMyPrizeList({
    openId: context.openId,
    activityId: context.data.activityId
  });
  return resultModel.success(result);
}
/**
 * 领取实物
 * @param {*} context 
 */
const receiveObjectPrize = async (context) => {
  // 判断活动
  let pipeRes = await pipeActivityInfo(context);
  // 活动结束可以领取实物
  if (pipeRes.code && pipeRes.code !== CODE_TYPES.ERROR_ACTIVITY_OVER.code) {
    return pipeRes;
  }
  let {
    provice,
    city,
    area,
    addressdetail,
    _id,
    name,
    phone,
    activityId
  } = context.data;
  if (!_id) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `缺少_id`)
  }
  if (!name) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `缺少name`)
  }
  if (!phone) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `缺少phone`)
  }
  if (!provice) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `缺少provice`)
  }
  if (!city) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `缺少city`)
  }
  if (!area) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `缺少area`)
  }
  if (!addressdetail) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `缺少addressdetail`)
  }
  let awardSer = new AwardsService(context);
  let result = await awardSer.recieveObjectPrize(_id, {
    activityId,
    provice,
    city,
    area,
    addressdetail,
    name,
    phone
  });
  if (result.code) {
    return resultModel.error(result);
  }
  return resultModel.success(result);
}

/**
 * 权益重新领取
 * @param {*} context 
 */
const receiveEnamePrize = async (context) => {
  let pipeRes = await pipeActivityInfo(context);
  // 活动结束可以领取权益
  if (pipeRes.code && pipeRes.code !== CODE_TYPES.ERROR_ACTIVITY_OVER.code) {
    return pipeRes;
  }
  let {
    _id
  } = context.data;
  if (!_id) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `缺少_id`)
  }
  let awardsSer = new AwardsService(context);
  let result = await awardsSer.recieveEnamePrize(_id, pipeRes._id);
  if (result.code) {
    return resultModel.error(result);
  }
  if (result.remark) {
    return resultModel.error(result, result.remark);
  }
  return resultModel.success(result);
}

const exchanagePrize = async (context) => {
  let {
    activityId,
    id
  } = context.data;
  let {
    openId
  } = context;
  // 参数校验
  if (!activityId) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `activityId必填`);
  }
  if (!id) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `id必填`);
  }
  try {
    const awardsSer = new AwardsService(context);
    const userSer = new UserService(context);
    let hasExchangeList = await awardsSer.getUserPrizeById(activityId, id);
    if (hasExchangeList.length) {
      resultModel.error(CODE_TYPES.ERROR_SYSTEM, '奖品已经兑换过了');
    }
    let prizeInfo = await awardsSer.getActivityPrizeById(id);
    if (!prizeInfo) return resultModel.error(CODE_TYPES.ERROR_SYSTEM, '奖品不存在');
    let {
      value,
      _id: prizeId,
      ...prizeRest
    } = prizeInfo;
    let {
      totalShoes,
      userNick,
      _id: userId
    } = await userSer.getUserInfo();
    if (totalShoes < value) {
      return resultModel.error(CODE_TYPES.ERROR_NO_SHOES);
    }
    let res = await awardsSer.reduceStock(prizeId);
    if (!res || res.code === CODE_TYPES.ERROR_NO_STOCK.code || res.code === CODE_TYPES.SYSTEM_ERROR.code) {
      return resultModel.error(CODE_TYPES.ERROR_EXCHANGE, res.defaultMsg || '兑换失败')
    }
    let shipType = prizeRest.type === PRIZE_TYPE.OBJECT ? {
      shipStatus: SHIP_TYPE.INIT
    } : {}
    let award = {
      prizeId,
      userNick,
      exchangeValue: value,
      drawStatus: DRAW_STATUS.WAITAWARD,
      ...prizeRest,
      ...shipType,
      openId,
    }
    let insertRes = await awardsSer.addAward(award);
    if (!insertRes) {
      return resultModel.error(CODE_TYPES.ERROR_EXCHANGE)
    }
    await userSer.updateUser(userId, {
      totalShoes: +totalShoes - +value
    })
    return resultModel.success({
      ...award,
      id: insertRes
    })
  } catch (e) {
    console.log(e, 'exchanagePrize')
    return resultModel.error(CODE_TYPES.ERROR_SYSTEM)
  }
}

const getExchanageList = async (context) => {
  let {
    activityId
  } = context.data;
  // 参数校验
  if (!activityId) {
    return resultModel.error(CODE_TYPES.PARAMS_ERROR, `activityId必填`);
  }
  try {
    const awardsSer = new AwardsService(context);
    const userSer = new UserService(context);
    let exchangePrizeList = await awardsSer.getPrizeByType(activityId, PRIZE_DATA_TYPE.EXCHANGE);
    let hasExchangePrize = await awardsSer.getUserPrizeByType(activityId, PRIZE_DATA_TYPE.EXCHANGE) || [];
    let hasEchangeIds = hasExchangePrize.map(i => i.prizeId);
    exchangePrizeList = exchangePrizeList.map(i => {
      if (hasEchangeIds.includes(i._id)) {
        return {
          ...i,
          status: HAS_EXCHANGE.EXCHANGE
        }
      }
      return {
        ...i,
        status: HAS_EXCHANGE.NO_EXCHANGE
      };
    })
    let {
      totalShoes
    } = await userSer.getUserInfo();
    return resultModel.success({
      totalShoes,
      list: exchangePrizeList
    })
  } catch (e) {
    console.log(e, 'getExchanageList')
    return resultModel.error(CODE_TYPES.ERROR_SYSTEM)
  }
}

module.exports = {
  getMyPrizeList,
  receiveObjectPrize,
  receiveEnamePrize,
  getExchanageList,
  exchanagePrize
}