/**
 * 用户相关方法
 */
const { BaseDao, TBAPIS } = require('../sdk');
const BaseService = require('./base.service');
const { USER_DB_NAME, DATA_DB_NAME } = require('../db');
const { getToday } = require('../utils')
const { TASK_TYPE_WIN } = require('../constants')

class UserService extends BaseService{
  constructor(context) {
    super(context);
    this.userdao = new BaseDao(context, USER_DB_NAME);
    this.dataDao = new BaseDao(context, DATA_DB_NAME);
  }
  /**
   * @desc 获取当前打开活动的用户详情
   * @returns 若用户不存在，返回null; 用户存在，返回用户信息(object对象)
   */
  async getUserInfo() {
    let { openId, data } = this.context;
    let { activityId } = data;
    let record = await this.userdao.findOne({openId, activityId });
    return record;
  }

  /**
   * @desc 根据inviteId获取用户详情
   * @desc 常用于助力分享码为用户openId, 被邀请人打开活动助力时需要获取邀请人的用户详情
   * @param {邀请人的openId} inviteId 
   * @returns 若用户不存在，返回null; 用户存在，返回用户信息(object对象)
   */
  async getUserInfoByOpenId(inviteId) {
    let { activityId } = this.context.data;
    let record = await this.userdao.findOne({openId: inviteId, activityId });
    return record;
  }

  /**
   * @desc 获取是否是会员
   * @param {调用淘宝接口的session} session 
   * @returns {isvip: boolean(是否是会员), url: string(入会链接) }
   */
  async getShopVip(session, callbackUrl = "") {
    // 会员mock，当传递的参数有vipmock: true，则表示已经是会员
    let { vipmock = false} = this.context.data;
    if (vipmock) {
      return {
        isvip: true
      }
    }
    let result = {}
    let shopUrl = {};
    try{
      result = await TBAPIS.queryVipinfo(this.context, session);
      shopUrl = await TBAPIS.getShopVipUrl(
        this.context, session, 
        {
          "source": "isvapp", "entrance": "duiba"
        },
        callbackUrl
      );
    }catch(e) {
      console.log(e);
    }
    console.log(`result, shopUrl`, result, shopUrl);
    return {
      isvip: !!(result.result && result.result.member_info),
      url: shopUrl.result && shopUrl.result.result
    }
  }

  /**
   * @desc 更新用户表
   * @param {用户的主键id} _id 
   * @param {更新的对象} document 
   * @returns 若更新成功，返回为1； 若更新失败，返回为 0 或系统直接报错
   */
  async updateUser(_id, document) {
    console.log(document);
    return await this.userdao.update({_id}, {$set: {
      ...document,
      updateTime: Date.now()
    }});
  }
  /**
   * @desc 根据用户主键id查找用户详情
   * @param {用户的主键id} _id 
   * @returns 若用户不存在，返回null; 用户存在，返回用户信息(object对象)
   */
  async getUserInfoById(_id) {
    return await this.userdao.findOne({_id});
  }

  async insertInitUser(userInfo) {
    let serverTime = Date.now();
    return this.userdao.insertOne({
      ...userInfo,
      createTime: serverTime,
      updateTime: serverTime,
      createDay: getToday(),
    });
  }

  /**
   * 查询用户积分
   */
  async queryUserCredits(session) {
    try {
      let result = await TBAPIS.queryCredits(this.context,session)
      console.log(result)
      return result;
    } catch (e) {
      console.log(`查询用户积分失败: `, e);
      return false;
    }
  }

  /**
   * 更新（加扣）用户积分
   * @param {number} quantity 数量
   * @param {string} optType add 增加 reduce 减少
   * @param {number} changeType 变更类型
   */
  async updateUserCredits(
    quantity,
    session,
    optType = 'reduce',
    changeType = 1
  ) {
    const isAdd = optType === 'add';
    try {
      const options = {
        quantity, // 积分数量, 每次变更的数量
        change_type: changeType, //变更类型：0交易，1：互动活动，2：权益兑换，3：手工调整
        opt_type: isAdd ? '0' : 1, // 操作类型，0：增加，1：扣减
        remark: `参与活动` // 备注
      };
      console.log(`${isAdd ? '加' : '扣'}积分请求参数`, options);
      const result = await TBAPIS.changeCredits(this.context,session,options)
      console.log(
        `${isAdd ? '加' : '扣'}积分返回结果:${JSON.stringify(result)}`
      );
      return result;
    } catch (e) {
      console.log(`更新用户积分失败`, e);
      return false;
    }
  }

  /**
   * 获取用户排名
   */
  async getUserRank(maxScore) {
    let { activityId } = this.context.data;
    const { openId } = this.context;
    let sameScoreList = await this.userdao.find({ maxScore: maxScore, activityId: activityId }, {
      sort: { maxScoreTime: 1 }
    });
    let rank = "";
    let gap = 0;
    //说明有多个跟自己同分数的人
    for (let j = 0; j < sameScoreList.length; j++) {
      if (sameScoreList[j].openId == openId) {
        gap = j;
      }
    }
    rank = await this.userdao.count({ maxScore: { $gt: maxScore }, activityId: activityId });
    console.log("rank", rank, "----", gap);
    rank = rank + 1 + gap;
    return { rank };
  }

  // 获取周边两位排名用户,及自己
  async getSideScoreUser(maxScore, myRank) {
    let { activityId } = this.context.data;
    let sameScoreList = await this.userdao.find({ maxScore: maxScore, activityId: activityId }, {
      sort: { maxScoreTime: 1 },
      projection: {
        userNick: 1,
        avatar: 1,
        maxScore: 1
      },
    });

    //防止概率性问题
    if(sameScoreList.length == 0) {
      sameScoreList = await this.userdao.find({ maxScore: maxScore, activityId: activityId }, {
        sort: { maxScoreTime: 1 },
        projection: {
          userNick: 1,
          avatar: 1,
          maxScore: 1
        },
      });
    }
    
    // 大于分数的一位
    const greaterUser = await this.userdao.find({ maxScore: { $gt: maxScore }, activityId: activityId }, {
      limit: 1,
      projection: {
        userNick: 1,
        avatar: 1,
        maxScore: 1
      },
      sort: {
        maxScore: 1,
        maxScoreTime: 1
      }
    });

    // 小于分数的一位
    const lessUser = await this.userdao.find({ maxScore: { $lt: maxScore }, activityId: activityId, maxScoreTime: { $exists: true } }, {
      limit: 1,
      projection: {
        userNick: 1,
        avatar: 1,
        maxScore: 1
      },
      sort: {
        maxScore: -1,
        maxScoreTime: 1
      }
    });

    // 没有和自己相同分数的人;
    console.log('sameScoreList'+JSON.stringify(sameScoreList))
    const noSameUser = sameScoreList.length === 1;
    const greaterUserInfo = greaterUser.length && greaterUser[0];
    const lessUserInfo = lessUser.length && lessUser[0];

    if(noSameUser) {
      let list = [];
      if(greaterUserInfo) {
        list.push({
          ...greaterUserInfo,
          rank: myRank - 1
        })
      };
      list.push({
        ...sameScoreList[0],
        isSelf: true,
        rank: myRank
      })
      if(lessUserInfo) {
        list.push({
          ...lessUserInfo,
          rank: myRank + 1
        })
      }
      return list.map(v => {
        if(v.isSelf) return v;
        return {
          ...v,
          userNick: `${v.userNick[0]}***${v.userNick[v.userNick.length -1]}`
        }
      })
    }

    let gap = 0;
    //说明有多个跟自己同分数的人
    for (let j = 0; j < sameScoreList.length; j++) {
      if (sameScoreList[j].openId == this.context.openId) {
        gap = j;
      }
    };

    // 多个分数相同的人 自己是第一个
    if(gap === 0) {
      let list = [];
      if(greaterUserInfo) {
        list.push({
          ...greaterUserInfo,
          rank: myRank - 1
        })
      };
      list.push({
        ...sameScoreList[0],
        isSelf: true,
        rank: myRank
      })
      list.push({
        ...sameScoreList[1],
        rank: myRank + 1
      })
      return list.map(v => {
        if(v.isSelf) return v;
        return {
          ...v,
          userNick: `${v.userNick[0]}***${v.userNick[v.userNick.length -1]}`
        }
      })
    }  // 多个分数相同的人 自己是最后一个
    else if (gap === sameScoreList.length - 1) {
      let list = [];
      list.push({
        ...sameScoreList[gap - 1],
        rank: myRank - 1
      })
      list.push({
        ...sameScoreList[gap],
        isSelf: true,
        rank: myRank
      })
      if(lessUserInfo) {
        list.push({
          ...lessUserInfo,
          rank: myRank + 1
        })
      }
      return list.map(v => {
        if(v.isSelf) return v;
        return {
          ...v,
          userNick: `${v.userNick[0]}***${v.userNick[v.userNick.length -1]}`
        }
      })
    } else {
      let list = [];
      list.push({
        ...sameScoreList[gap - 1],
        rank: myRank - 1
      })
      list.push({
        ...sameScoreList[gap],
        isSelf: true,
        rank: myRank
      })
      list.push({
        ...sameScoreList[gap + 1],
        rank: myRank + 1
      })
      return list.map(v => {
        if(v.isSelf) return v;
        return {
          ...v,
          userNick: `${v.userNick[0]}***${v.userNick[v.userNick.length -1]}`
        }
      })
    }
  }

  async getTotalRankUserCount(activityId) {
    return await this.userdao.count({ activityId, maxScoreTime: { $exists: true } });
  }

  // 排行榜, 前1000名
  async getRankList(activityId, pageNum, pageSize) {
    console.log(`activityId`, activityId);
    return await this.userdao.find({ activityId, maxScoreTime: { $exists: true } }, {
      projection: {
        userNick: 1,
        avatar: 1,
        totalScore: 1,
        openId: 1
      },
      sort: {
        totalScore: -1,
        maxScoreTime: 1
      },
      limit: pageSize,
      skip: (pageNum - 1) * pageSize,
    });
  }
}

module.exports = UserService;