/** @format */

import { CODE_TYPES, EIGHT_HOURS } from './../constants'
import { services, paramsCheck } from '../decorator'

import { resultsModel } from '../sdk'
import { formatTime } from '../utils'
import { AwardsService } from '../service'
import * as xlsx from 'node-xlsx'
//奖池类型
const prizePoolType = {
  '1': '排行榜奖励',
  '2': '兑换奖品',
  '3': '大转盘奖品',
  '4': '其他奖励'
}
//发货状态
const shipMsg = {
  '1': '未填地址',
  '2': '未发货',
  '3': '已发货'
}

const prizeKind = {
  '1': '优惠券',
  '2': '权益',
  '3': '实物'
}

export default class BaseController {
  /**
   *  活动中奖名单
   */
  @paramsCheck(['activityId'])
  @services([AwardsService])
  async findWinnerInfoList(context: Context, { session }, [awardsService]: [AwardsService]) {
    const { activityId, pageNo = 1, pageSize = 500 } = context.data
    // 获取活动中奖列表
    let awardslist = await awardsService.getAwardslistByActivityId(activityId, pageNo, pageSize)

    console.log(`awardslist: ${JSON.stringify(awardslist)}`)

    let winnersObj = {}
    awardslist.forEach(v => {
      if (!winnersObj[v.rank]) {
        winnersObj[v.rank] = []
      }
      winnersObj[v.rank].push({
        userNick: v.userNick,
        id: v.openId
      })
    })

    console.log(`winnersObj: ${JSON.stringify(winnersObj)}`)

    let results = []
    new Map(Object.entries(winnersObj)).forEach((v, index) => {
      results.push({
        rank: index,
        winnerDetailList: v
      })
    })
    return resultsModel.success(results)
  }

  /**
   *  导出活动中奖名单
   */
  @paramsCheck(['activityId', 'title'])
  @services([AwardsService])
  async exportAwardsList(context: Context, { session }, [awardsService]: [AwardsService]) {
    const { cloud } = context
    const { activityId, title } = context.data
    // 获取活动中奖列表
    let pageNo = 1
    let pageSize = 500

    let awardslist = []
    // 自动分页
    while (awardslist.length >= (pageNo - 1) * 500) {
      let nowPageNoList = await awardsService.getAwardslistByActivityId(activityId, pageNo, pageSize)
      awardslist = [...awardslist, ...nowPageNoList]
      pageNo += 1
    }
    console.log(`awardslist: ${JSON.stringify(awardslist)}`)

    let xlsxData = [['序列', '名次', '昵称', '中奖名称']]
    awardslist.forEach((v, index, arr) => {
      xlsxData.push([index + 1, v.rank, v.userNick, v.name])
    })
    let buffer = xlsx.build([
      {
        name: title + new Date().getTime(),
        data: xlsxData
      }
    ])
    console.log(`xlsxData: ${JSON.stringify(xlsxData)}`)
    let result = await cloud.file.uploadFile({
      fileContent: buffer,
      fileName: title + new Date().getTime() + '.xlsx'
    })
    // result.url 需进行处理
    if (result.url) {
      result.url = result.url.replace('http', 'https').replace('-internal', '')
    }
    return resultsModel.success(result)
  }

  @services([AwardsService])
  async exportAwardsInfoBySearch(context: Context, { session }, [awardsService]: [AwardsService]) {
    /******VV */
    let { cloud } = context
    let { activityId, title, prizeId, userId: userNick, timeStart, timeEnd, shipStatus, prizeDataType } = context.data
    if (!activityId) {
      return resultsModel.error(CODE_TYPES.PARAMS_ERROR, `缺少activityId`)
    }
    if (!title) {
      return resultsModel.error(CODE_TYPES.PARAMS_ERROR, `缺少title`)
    }
    console.log('prizeId=====', prizeId)
    if (!prizeId && !userNick && !timeStart && !timeEnd && !shipStatus && !prizeDataType) {
      return resultsModel.error(CODE_TYPES.PARAMS_ERROR, `未填写筛选信息`)
    }
    console.log(
      'activityId--1--',
      activityId,
      'prizeId--1-',
      prizeId,
      'userNick--1-',
      userNick,
      'timeStart--1-',
      timeStart,
      'timeEnd--1-',
      timeEnd
    )

    // 获取活动中奖列表
    let total = await awardsService.getTotalObjectCount(
      activityId,
      prizeId,
      userNick,
      timeStart,
      timeEnd,
      shipStatus,
      prizeDataType
    )
    let awardslist = []
    //导出中奖名单的pagesize和pageNo
    let pageSize0 = 500
    let pageNo = 1
    if (total > pageSize0) {
      let totalPage = Math.ceil(total / pageSize0)
      for (let i = 1; i <= totalPage; i++) {
        pageNo = i
        let awardMore = await awardsService.getUserPrizeByActivityId(
          activityId,
          pageNo,
          pageSize0,
          prizeId,
          userNick,
          timeStart,
          timeEnd,
          shipStatus,
          prizeDataType
        )
        awardslist = awardslist.concat(awardMore)
      }
    } else {
      awardslist = await awardsService.getUserPrizeByActivityId(
        activityId,
        pageNo,
        total,
        prizeId,
        userNick,
        timeStart,
        timeEnd,
        shipStatus,
        prizeDataType
      )
    }
    console.log(`awardslist:》》》 ${JSON.stringify(awardslist)}《《《`)
    if (awardslist.length > 0) {
      let xlsxData = [
        [
          '序列',
          '奖品编号',
          '中奖时间',
          '地址填写时间',
          '用户昵称',
          '奖品种类',
          '奖品类型',
          '中奖名称',
          '收货人',
          '手机号',
          '省',
          '市',
          '区',
          '详细地址',
          '发货状态',
          '物流公司',
          '物流单号'
        ]
      ]
      awardslist.forEach((v, index, arr) => {
        xlsxData.push([
          index + 1,
          v._id,
          //中奖时间-填写地址时间
          v.createTime ? formatTime(new Date(v.createTime + EIGHT_HOURS)) : '',
          v.type == 1 ? '/' : v.receiveTime ? formatTime(new Date(v.receiveTime + EIGHT_HOURS)) : '未填',
          v.userNick || '/',
          prizePoolType[v.prizeDataType],
          prizeKind[v.type],
          v.name || '',
          v.type == 1 ? '/' : v.receiveName || '',
          v.type == 1 ? '/' : v.phone || '未填',
          v.type == 1 ? '/' : v.province ? v.province: '未填',
          v.type == 1 ? '/' : v.city ? v.city : '未填',
          v.type == 1 ? '/' : v.area ? v.area : '未填',
          v.type == 1 ? '/' : v.addressDetail ? v.addressDetail : '未填',
          v.type == 1 ? '/' : shipMsg[v.shipStatus],
          v.type == 1 ? '/' : v.shipCompany || '/',
          v.type == 1 ? '/' : v.shipNum || '/'
        ])
      })
      let buffer = xlsx.build([
        {
          name: title + new Date().getTime(),
          data: xlsxData
        }
      ])
      try {
        let result = await cloud.file.uploadFile({
          fileContent: buffer,
          fileName: title + new Date().getTime() + '.xlsx'
        })
        // result.url 需进行处理
        if (result.url) {
          result.url = result.url.replace('http', 'https').replace('-internal', '')
        }
        return resultsModel.success(result)
      } catch (e) {
        console.log('上传文件出错', e)
        // 打印日志
        return resultsModel.error(CODE_TYPES.SYSTEM_ERROR, `上传文件错误`)
      }
    } else {
      return resultsModel.success(false)
    }
  }

  //筛选中奖名单
  @services([AwardsService])
  async findWinnerInfoListBySearch(context: Context, { session }, [awardsService]: [AwardsService]) {
    let { cloud } = context
    let {
      activityId,
      title,
      pageNo,
      pageSize,
      prizeId,
      userId: userNick,
      timeStart,
      timeEnd,
      shipStatus,
      prizeDataType
    } = context.data
    console.log('data>>>>>', context.data)
    if (!activityId) {
      return resultsModel.error(CODE_TYPES.PARAMS_ERROR, `缺少activityId`)
    }
    if (!title) {
      return resultsModel.error(CODE_TYPES.PARAMS_ERROR, `缺少title`)
    }
    console.log('prizeId=====', prizeId)
    if (!prizeId && !userNick && !timeStart && !timeEnd && !shipStatus && !prizeDataType) {
      return resultsModel.error(CODE_TYPES.PARAMS_ERROR, `未填写筛选信息`)
    }
    console.log(
      'activityId----',
      activityId,
      'prizeId---',
      prizeId,
      'userNick---',
      userNick,
      'timeStart---',
      timeStart,
      'timeEnd---',
      timeEnd,
      shipStatus
    )
    // 获取活动中奖列表
    let total = await awardsService.getTotalObjectCount(
      activityId,
      prizeId,
      userNick,
      timeStart,
      timeEnd,
      shipStatus,
      prizeDataType
    )
    console.log('totle=========', total)
    let awardslist = []
    if (total > 10) {
      let totalPage = Math.ceil(total / pageSize)
      let awardMore = await awardsService.getUserPrizeByActivityId(
        activityId,
        pageNo,
        pageSize,
        prizeId,
        userNick,
        timeStart,
        timeEnd,
        shipStatus,
        prizeDataType
      )
      // console.list()
      awardslist = awardslist.concat(awardMore)
    } else {
      awardslist = await awardsService.getUserPrizeByActivityId(
        activityId,
        pageNo,
        pageSize,
        prizeId,
        userNick,
        timeStart,
        timeEnd,
        shipStatus,
        prizeDataType
      )
      console.log('awardslist>>>>', awardslist)
    }
    console.log(`awardslist:》》》 ${JSON.stringify(awardslist)}《《《`)
    awardslist.map(item => {
      let ad = item.province + item.city + item.area + item.addressDetail,
        type0 = item.type
      item.prizeId = item._id
      item.createTime = formatTime(new Date(item.createTime + EIGHT_HOURS)) || ''
      item.receiveTime = item.receiveTime ? formatTime(new Date(item.receiveTime + EIGHT_HOURS)) : '未填'
      item.prizeDataType = prizePoolType[item.prizeDataType]
      item.type = prizeKind[item.type]
      item.receiveName = type0 == 3 ? item.receiveName || '未填' : '/'
      item.phone = type0 == 3 ? item.phone || '未填' : '/'
      item.address = type0 == 3 ? (item.province ? ad : '未填') : '/'
      item.shipStatus = type0 == 3 ? shipMsg[item.shipStatus] : '/'
      item.shipCompany = type0 == 3 ? item.shipCompany || '/' : '/'
      item.shipNum = type0 == 3 ? item.shipNum || '/' : '/'
    })
    // console.log(`xlsxData: ${JSON.stringify(xlsxData)}`);
    try {
      return resultsModel.success({ list: awardslist, total })
    } catch (e) {
      console.log('上传文件出错', e)
      // 打印日志
      return resultsModel.error(CODE_TYPES.SYSTEM_ERROR, `查看结果错误`)
    }
  }

  /*
[{
  "data": [
    ["序列", "中奖时间", "收获地址", "邮编"],
    ["1","creatTime","XXX","XXX"]
  ],
  "name": "测试活动LSS1594797305510"
}]
*/
  //导入发货状态信息
  @services([AwardsService])
  async downloadFile(context: Context, { session }, [awardsService]: [AwardsService]) {
    const cloud = context.cloud
    //获取活动id
    const { activityId, fileName } = context.data
    console.log('====--=-=-=-=-', fileName)
    //-----------
    let fileId = fileName
    let result,
      total = 0
    try {
      result = await cloud.file.downloadFile({ fileId })
      const sheets = xlsx.parse(result)
      //接口调用
      let { data } = sheets[0]
      if (data[1].length == 3) {
        for (let i = 1; i < data.length; i++) {
          //activityId, prizeId, shipCompany, shipNum,
          let prizeId = data[i][0]
          let shipCompany = data[i][1]
          let shipNum = data[i][2]
          let result2 = await awardsService.setUserShipStatus(activityId, prizeId, shipCompany, shipNum)
          total += result2
          console.log('>>>')
        }
        return resultsModel.success({ total, data })
      } else {
        return resultsModel.error(CODE_TYPES.SYSTEM_ERROR, '导入文件格式错误')
      }
    } catch (e) {
      console.log('e', e)
      return resultsModel.error(CODE_TYPES.SYSTEM_ERROR, '导入文件出现错误')
    }
  }
}
