/**
 * 基本信息
 *
 * @format
 */

import { BaseDao, dateFormatter, transformBeijingDate } from '../../sdk'
import { STAT_DB_NAME, ACCESS_DB_NAME, USER_DB_NAME, AWARDS_DB_NAME, JOIN_DB_NAME } from '../../db'
import * as xlsx from 'node-xlsx'
import { getToday } from '../../utils'
import { STAT_TYPE } from '../../constants'
import { formatDate } from '../../utils/common/date'

export default class StatService {
  context: IContext<any>
  statdao: IBaseDao
  accessdao: IBaseDao
  userdao: IBaseDao
  awardsdao: IBaseDao
  joindao: IBaseDao
  constructor(context: IContext<IParams>) {
    this.context = context
    this.statdao = new BaseDao(context, STAT_DB_NAME)
    this.accessdao = new BaseDao(context, ACCESS_DB_NAME)
    this.userdao = new BaseDao(context, USER_DB_NAME)
    this.joindao = new BaseDao(context, JOIN_DB_NAME)
    this.awardsdao = new BaseDao(context, AWARDS_DB_NAME)
  }

  /**
   *  添加数据统计记录
   *
   * @param {number} type  数据类型
   * @param {object} [customStatData={}] 自定义数据
   * @param {IUserInfo} [userInfo] 用户信息
   * @return {boolean}
   * @memberof StatService
   */
  async addStat(
    type: number,
    userInfo: IUserInfo = {} as IUserInfo,
    customStatData: object = {}
  ): Promise<{ id: string }> {
    const { openId } = this.context
    const { activityId } = this.context.data
    const { userNick } = userInfo
    const id = await this.statdao.insertOne({
      activityId,
      type,
      typeName: STAT_TYPE[type],
      userNick,
      openId,
      ...customStatData,
      createTime: Date.now(),
      createDay: getToday()
    })
    return { id }
  }

  async getStats(activityId: string, startDay: string, endDay: string): Promise<any> {
    let day = startDay
    const oneDay = 24 * 3600 * 1000
    let xlsxData = []
    let curDayTimestamp = formatDate(day).getTimestamp()
    const endDayTimestamp = formatDate(endDay).getTimestamp()
    while (curDayTimestamp <= endDayTimestamp) {
      // 访问PV
      const PV = await this.accessdao.count({ activityId, createDay: day })
      // 访问UV
      const UV = await this.userdao.count({ activityId, [`login.${day}`]: { $exists: true } })
      // 新增UV
      const newUV = await this.userdao.count({ activityId, createDay: day })
      // 新增PV
      // const newPV = await this.userdao.count({ activityId, createDay: day })

      // 新增UV(通过邀请)
      const newUVFromInviteUV = (
        await this.accessdao.aggregate([
          { $match: { activityId, createDay: day, inviteId: { $exists: true } } },
          { $project: { openId: true } },
          { $group: { _id: '$openId', count: { $sum: 1 } } }
        ])
      ).length

      // 已入会(老会员)PV
      const vipPV = await this.accessdao.count({
        activityId,
        createDay: day,
        //@ts-ignore
        $or: [{ 'member.flag': true }, { 'member.newMember': true }],
        'member.bememberDay': { $not: { $eq: day } }
      })

      // 已入会(老会员)UV
      const vipUV = await this.userdao.count({
        activityId,
        [`login.${day}`]: { $exists: true },
        $or: [
          //@ts-ignore
          { 'member.flag': true },
          //@ts-ignore
          { 'member.newMember': true, 'member.bememberTime': { $lt: new Date(day).getTime() } }
        ]
      })

      // 未入会PV
      const noVipPV = await this.accessdao.count({ activityId, createDay: day, isVip: false })

      // 未入会UV
      const noVipUV = (
        await this.accessdao.aggregate([
          { $match: { activityId, createDay: day, isVip: false } },
          { $project: { openId: true } },
          { $group: { _id: '$openId', count: { $sum: 1 } } }
        ])
      ).length

      // 新增入会UV
      const newVipUV = await this.userdao.count({ activityId, 'member.newMember': true, 'member.bememberDay': day })

      const newFollowUV = await this.userdao.count({ activityId, 'follow.newFollow': true, 'follow.followDay': day })

      // 助力成功UV
      const helpSuccessUV = (
        await this.statdao.aggregate([
          { $match: { activityId, createDay: day, type: STAT_TYPE.INITE_SUCCESS } },
          { $project: { openId: true } },
          { $group: { _id: '$openId', count: { $sum: 1 } } }
        ])
      ).length

      // 根据任务类型获取完成任务的人数
      //  example: await getTaskCompleteUV('collectGoods', day)
      const getTaskCompleteUV = async (task: string, day: string) => {
        return await this.userdao.count({
          activityId,
          [`taskInfo.${day}.${task}`]: { $exists: true }
        })
      }

      const keyValueMapper = {
        时间: day,
        访问PV: PV,
        访问UV: UV,
        新增UV: newUV,
        '新增UV(通过邀请)': newUVFromInviteUV,
        '已入会(老会员)PV': vipPV,
        '已入会(老会员)UV': vipUV,
        未入会PV: noVipPV,
        未入会UV: noVipUV,
        新入会UV: newVipUV,
        新增关注UV: newFollowUV,
        助力成功UV: helpSuccessUV
        //收藏商品任务完成UV: await getTaskCompleteUV('collectGoods', day)
      }
      console.log(keyValueMapper, 'xlsxData')

      if (day === startDay) {
        xlsxData.push(Object.keys(keyValueMapper))
      }
      // @ts-ignore
      xlsxData.push(Object.values(keyValueMapper))
      const tommorrow = new Date(day).getTime() + oneDay
      curDayTimestamp = tommorrow
      day = dateFormatter(transformBeijingDate(tommorrow), 'yyyy/MM/dd')
    }

    return xlsxData
  }

  async buildExcel(startDay: string, endDay: string, xlsxData: []) {
    let buffer = xlsx.build([
      {
        name: `数据统计${startDay.replace(/\//g, '-')}-${endDay.replace(/\//g, '-')}`,
        data: xlsxData
      }
    ])

    let result = await this.context.cloud.file.uploadFile({
      fileContent: buffer,
      fileName: `数据统计${startDay.replace(/\//g, '-')}-${endDay.replace(/\//g, '-')}.xlsx`
    })
    if (result.url) {
      result.url = result.url.replace('http', 'https').replace('-internal', '')
    }
    return result
  }

  async getStatCountByquery(query: IMongoQuery) {
    return await this.statdao.count(query)
  }
}
