/**
 * 通用toast
 * @param {string} content toast内容
 * @param {number} duration 显示时长
 * @param {function} successCb 成功回调
 */
export const commonToast = (content, duration = 3000, successCb) => {
  my.showToast({
    content: content || '换个姿势再试一次~',
    duration,
    success: successCb
  });
};

/**
 * 简易alert
 * @param {string} content toast内容
 */
export const simpleAlert = content => {
  my.alert({
    content
  });
};

/**
 * 通用确认弹窗
 * @param {string} title 标题
 * @param {string} content 内容
 * @param {function} successCb 成功回调
 */
export const commonConfirm = (title = '提示', content, successCb) => {
  my.confirm({
    title,
    content,
    confirmButtonText: '确定',
    cancelButtonText: '取消',
    success: result => {
      const { confirm } = result;
      if (confirm) {
        successCb && successCb();
      }
    }
  });
};

/**
 * 获取授权用户信息
 * @returns {object} 用户登录信息
 */
export const getAuthUserInfo = async () => {
  return new Promise((resolve, reject) => {
    my.authorize({
      scopes: 'scope.userInfo',
      success: () => {
        my.getAuthUserInfo({
          success: userInfo => {
            resolve(userInfo);
          },
          fail: err => {
            reject(err);
          }
        });
      },
      fail: err => {
        reject(err);
      }
    });
  });
};

/**
 * 获取系统信息
 * @returns {object} 小程序系统信息
 */
export const getSystemInfo = async () => {
  return new Promise((resolve, reject) => {
    my.getSystemInfo({
      success: res => {
        resolve(res);
      },
      fail: err => {
        reject(err);
      }
    });
  });
};

/**
 * 执行关注店铺
 * @param {number} sellerId 店铺归属的卖家Id
 * @returns {boolean} 关注状态
 */
export const favorShop = async sellerId => {
  if (!sellerId) {
    console.error('favorShop: 请传入卖家Id');
    return false;
  }
  return new Promise((resolve, reject) => {
    my.tb.favorShop({
      id: +sellerId,
      success: res => {
        resolve(res);
      },
      fail: err => {
        if (err.error === 11) {
          // error code 11 为用户取消操作不认作失败场景
          resolve(false);
        }
        reject(err);
      }
    });
  });
};

/**
 * 判断是否关注店铺
 * @param {number} sellerId 店铺归属的卖家Id
 * @returns {boolean} 关注状态
 */
export const checkShopFavoredStatus = async sellerId => {
  if (!sellerId) {
    console.error('checkShopFavoredStatus: 请传入卖家Id');
    return false;
  }
  return new Promise((resolve, reject) => {
    my.tb.checkShopFavoredStatus({
      id: +sellerId,
      success: res => {
        resolve(res.isFavor);
      },
      fail: err => {
        reject(err);
      }
    });
  });
};

/**
 * 跳转到外部链接
 * @param {string} url 跳转链接
 */
export const navigateToOutside = url => {
  if (!url) {
    console.error('navigateToOutside: 请传入url');
    return false;
  }
  my.call('navigateToOutside', { url });
};

/**
 * 跳转到内部链接（新开窗口）
 * @param {string} url 跳转链接
 */
export const navigateTo = url => {
  if (!url) {
    console.error('navigateTo: 请传入url');
    return false;
  }
  my.navigateTo({ url });
};

/**
 * 跳转到内部链接（不新开窗口）
 * @param {string} url 跳转链接
 */
export const redirectTo = url => {
  if (!url) {
    console.error('redirectTo: 请传入url');
    return false;
  }
  my.redirectTo({ url });
};

/**
 * 获取服务器时间
 * @returns {number} 服务器时间戳
 */
export const getServerTime = async () => {
  return new Promise((resolve, reject) => {
    my.getServerTime({
      success: res => {
        resolve(res.time);
      },
      fail: err => {
        reject(err);
      }
    });
  });
};

/**
 * 收藏商品
 * @param {number} 商品id
 * @returns {object} 收藏结果
 */
export const collectGoods = async id => {
  if (!id) {
    console.error('collectGoods: 请传入商品id');
    return false;
  }
  return new Promise((resolve, reject) => {
    my.tb.collectGoods({
      id: +id,
      success: res => {
        resolve(res);
      },
      fail: err => {
        if (err.error === 11) {
          // error code 11 为用户取消操作不认作失败场景
          resolve(false);
        }
        reject(err);
      }
    });
  });
};

/**
 * 查询商品收藏状态
 * @param {number} 商品id
 * @returns {boolean} 商品收场状态
 */
export const checkGoodsCollectedStatus = async id => {
  if (!id) {
    console.error('checkGoodsCollectedStatus: 请传入商品id');
    return false;
  }
  return new Promise(resolve => {
    my.tb.checkGoodsCollectedStatus({
      id: +id,
      success: res => {
        resolve(res.isCollect);
      }
    });
  });
};

/**
 * 跳转到淘宝商品页
 * @param {string} itemId 商品ID
 */
export const openDetail = async itemId => {
  if (!itemId) {
    console.error('openDetail: 请传入商品id');
    return false;
  }
  return new Promise((resolve, reject) => {
    my.tb.openDetail({
      itemId: itemId + '',
      success: res => {
        resolve(res);
      },
      fail: res => {
        reject(res);
      }
    });
  });
};

/**
 * 创建音频类
 */
class AudioModel {
  constructor(opts = {}) {
    const { src } = opts;
    if (!src) {
      console.error('请输入音频链接');
      return false;
    }
    try {
      const innerAudioContext = my.createInnerAudioContext();
      Object.keys(opts).forEach(item => {
        innerAudioContext[item] = opts[item];
      });
      return innerAudioContext;
    } catch (e) {
      console.error('当前版本不支持音频');
    }
  }
}
/**
 * 创建音频实例
 * @param {object} opts 配置对象 例 { src: '音频链接', autoplay: false }
 */
export const createAudio = opts => {
  const audioInstance = new AudioModel(opts);
  return audioInstance;
};

/**
 * 获取淘宝用户收货地址
 */
export const getUserAddress = async (opts = {}) => {
  let defaults = {
    addAddress: 'show',
    searchAddress: 'hide',
    locateAddress: 'hide',
    joinTag: '' // 分割收货地址符号（淘宝商家有特定发货需求）
  };
  let _opts = Object.assign({}, defaults, opts);
  return new Promise((resolve, reject) => {
    my.authorize({
      scopes: 'scope.addressList',
      success: () => {
        const { addAddress, searchAddress, locateAddress, joinTag } = _opts;
        if (!my.tb.chooseAddress) {
          reject({
            noSupport: true,
            errorMessage: '当前版本不支持选择收货地址，请升级到最新版本'
          });
          return false;
        }
        my.tb.chooseAddress(
          {
            addAddress,
            searchAddress,
            locateAddress
          },
          res => {
            const { error, provinceName, countyName, cityName, streetName, detailInfo, name, telNumber } = res;
            // 如果有error为用户取消操作不做操作
            if (error) {
              resolve(false);
            } else {
              resolve({
                ...res,
                duibaAddress: {
                  name,
                  phone: telNumber,
                  address: [provinceName, cityName, countyName, streetName, detailInfo].join(joinTag)
                }
              });
            }
          },
          res => {
            reject(res);
          }
        );
      }
    });
  });
};

/**
 * 选择图片
 */
export const chooseImage = async (opts = {}) => {
  let defaults = {
    count: 1,
    sourceType: ['camera', 'album'],
    compressLevel: 1
  };
  let _opts = Object.assign({}, defaults, opts);
  const { count, sourceType, compressLevel } = _opts;
  return new Promise((resolve, reject) => {
    if (!my.chooseImage) {
      reject({
        noSupport: true,
        errorMessage: '当前版本不支持选择收货地址，请升级到最新版本'
      });
      return false;
    }
    my.chooseImage({
      count,
      sourceType,
      success: res => {
        my.compressImage({
          apFilePaths: res.apFilePaths,
          level: compressLevel,
          success: data => {
            resolve(data);
          },
          fail: err => {
            resolve(err);
          }
        });
      },
      fail: err => {
        // error code 2001 为摄像头授权取消、ios 11为用户取消操作不认作失败场景 、安卓 为1
        if (err.error !== 2001 && err.error !== 11 && err.error !== 1) {
          reject(err);
        } else {
          resolve(false);
        }
      }
    });
  });
};

/**
 * 获取图片信息
 * @param {string} src 图片地址
 */
export const getImageInfo = async src => {
  return new Promise((resolve, reject) => {
    my.getImageInfo({
      src,
      success: res => {
        resolve(res);
      },
      fail: err => {
        reject(err);
      }
    });
  });
};

/**
 * 获取图片信息
 * @param {*} opts chooseImage配置参数
 */
export const uploadImage = async (opts = {}) => {
  const { cloud } = getApp();
  if (!cloud) {
    console.error('chattingSendCard: 请传入app.cloud');
    return false;
  }
  const chooseImageResult = await chooseImage(opts).catch(err => {
    commonToast(err && (err.errorMessage || '选择图片失败'));
  });
  if (chooseImageResult) {
    const { apFilePaths } = chooseImageResult;
    const filePath = apFilePaths[0];
    const imageInfo = await getImageInfo(filePath).catch(err => {
      commonToast(err && err.errorMessage);
    });
    // 找不到图片信息
    if (!imageInfo) {
      return false;
    }
    const { path: imagePath } = imageInfo;
    const { url } = await cloud.file.uploadFile({
      filePath,
      fileType: 'image',
      fileName: imagePath.split('/').pop()
    });
    return url;
  }
  return false;
}

/**
 * 分享到群聊卡片
 * @param {object} requestInfo 分享到群聊卡片所需请求参数
 */
export const chattingSendCard = async (requestInfo) => {
  const { cloud } = getApp();
  if (!cloud) {
    console.error('chattingSendCard: 请传入app.cloud');
    return false;
  }

  if (!requestInfo) {
    console.error('chattingSendCard: 请传入卡片请求参数');
    return false;
  }
  return new Promise((resolve, reject) => {
    my.authorize({
      scopes: 'scope.groupChat',
      success: async () => {
        const result = await cloud.topApi
          .invoke({
            api: 'taobao.chatting.platform.message.sendcard',
            data: { request_info: JSON.stringify(requestInfo) }
          })
          .catch(err => {
            reject(err);
          });
        if (result) {
          const { error } = result;
          if (error) {
            resolve(false);
          } else {
            resolve(result);
          }
        }
      },
      fail: err => {
        if (err.error === 11) {
          // error code 11 为用户取消操作不认作失败场景
          resolve(false);
        }
        reject(err);
      }
    });
  });
};

/**
 * 分享到群聊卡片
 * @param {*} cloud 小程序云
 * @param {object} requestInfo 分享到群聊卡片所需请求参数
 */
export const textRiskIdentification = async (text) => {
  return new Promise((resolve, reject) => {
    if (!my.tb.textRiskIdentification) {
      reject({
        noSupport: true,
        errorMessage: '不支持文本校验'
      });
    }
    my.tb.textRiskIdentification({
      data:{
        text
      },
      success: res => {
        if (res.data && res.data.result) {
          const { suggestion }= res.data.result;
          if (suggestion === 'pass') {
            resolve(true);
          }
        }
        resolve(false);
      },
      fail: err => {
        reject(err)
      },  
    })
  })
}

// 剪切板
export const setClipboard = async (params) => {
	return await promisifyMyApi('setClipboard', params)
};


function promisifyMyApi(methodName, params = {}) {
	return new Promise((resolve, reject) => {
		my[methodName]({
			...params,
			success: res => {
				resolve(res)
			},
			fail: err => {
				reject(err)
			}
		})
	})
}
