import { Camera, Matrix4, Object3D, Scene, Vector3 } from 'three'

type CSS2DObjctInfer = {
  element?: HTMLElement,
  isCSS2DObject?: boolean,
} & Object3D

export class CSS2DObjct extends Object3D {
  isCSS2DObject: boolean = true
  element: HTMLElement
  constructor(element: HTMLElement=document.createElement('div'), style?: Partial<CSSStyleDeclaration>, attributes?: any) {
    super()
    this.element = element
    this.element.style.position = 'absolute'
    this.element.style.userSelect = 'none'
    this.element.setAttribute('draggle', 'false')

    if (style) {
      for (var k in style) {
        this.element.style[k] = style[k]
      }
    }

		if (attributes) {
      for (var k in attributes) {
				// @ts-expect-error
        this.element[k] = attributes[k]
      }
    }

    this.addEventListener('removed', () => {
      this.traverse(function(obj: any) {
        if (obj.element instanceof Element && obj.element.parentNode !== null) {
          obj.element.parentNode.removeChild(obj.element)
        }
      })
    })
  }

  copy(source: this, recursive: boolean) {
    super.copy(source, recursive)
    // @ts-expect-error
    this.element = source.element.cloneNode(true)
    return this
  }
}

const _vector = new Vector3();
const _viewMatrix = new Matrix4();
const _viewProjectionMatrix = new Matrix4();
const _a = new Vector3();
const _b = new Vector3();

interface CSS2DRendererProps {
  element?: HTMLElement
}

export class CSS2DRenderer {
  domElement: HTMLElement
  render: (scene: any, camera: any) => void
  setSize: (width: any, height: any) => void

	constructor( parameters: CSS2DRendererProps = {} ) {

		const _this = this;

		let _width: number, _height: number;
		let _widthHalf: number, _heightHalf: number;

		const cache = {
			objects: new WeakMap()
		};

		const domElement = parameters.element !== undefined ? parameters.element : document.createElement( 'div' );

		domElement.style.overflow = 'hidden';
		domElement.style.pointerEvents = 'none'

		this.domElement = domElement;

		// this.getSize = function () {

		// 	return {
		// 		width: _width,
		// 		height: _height
		// 	};

		// };

		this.render = function ( scene, camera ) {

			if ( scene.autoUpdate === true ) scene.updateMatrixWorld();
			if ( camera.parent === null ) camera.updateMatrixWorld();

			_viewMatrix.copy( camera.matrixWorldInverse );
			_viewProjectionMatrix.multiplyMatrices( camera.projectionMatrix, _viewMatrix );

			renderObject( scene, scene, camera );
			zOrder( scene );

		};

		this.setSize = function ( width, height ) {

			_width = width;
			_height = height;

			_widthHalf = _width / 2;
			_heightHalf = _height / 2;

			domElement.style.width = width + 'px';
			domElement.style.height = height + 'px';

		};

		function renderObject( object: CSS2DObjctInfer, scene: Scene, camera: Camera ) {

			if ( object.isCSS2DObject ) {
				// console.log(object)
				// object.onBeforeRender( _this, scene, camera );

				_vector.setFromMatrixPosition( object.matrixWorld );
				_vector.applyMatrix4( _viewProjectionMatrix );

				const element = object.element;

				if ( /apple/i.test( navigator.vendor ) ) {

					// https://github.com/mrdoob/three.js/issues/21415
					element.style.transform = 'translate(-50%,-50%) translate(' + Math.round( _vector.x * _widthHalf + _widthHalf ) + 'px,' + Math.round( - _vector.y * _heightHalf + _heightHalf ) + 'px)';

				} else {
					element.style.transform = 'translate(-50%,-50%) translate(' + ( _vector.x * _widthHalf + _widthHalf ) + 'px,' + ( - _vector.y * _heightHalf + _heightHalf ) + 'px)';

				}

				element.style.display = ( object.visible && _vector.z >= - 1 && _vector.z <= 1 ) ? '' : 'none';

				const objectData = {
					distanceToCameraSquared: getDistanceToSquared( camera, object )
				};

				cache.objects.set( object, objectData );

				if ( element.parentNode !== domElement ) {

					domElement.appendChild( element );

				}

				// object.onAfterRender( _this, scene, camera );

			}

			for ( let i = 0, l = object.children.length; i < l; i ++ ) {
				renderObject( object.children[i], scene, camera );

			}

		}

		function getDistanceToSquared( object1: Camera, object2: CSS2DObjctInfer ) {

			_a.setFromMatrixPosition( object1.matrixWorld );
			_b.setFromMatrixPosition( object2.matrixWorld );

			return _a.distanceToSquared( _b );

		}

		function filterAndFlatten( scene: { traverse: (arg0: (object: any) => void) => void } ) {

			const result: any[] = [];

			scene.traverse( function ( object: { isCSS2DObject: any } ) {

				if ( object.isCSS2DObject ) result.push( object );

			} );

			return result;

		}

		function zOrder( scene: any ) {

			const sorted = filterAndFlatten( scene ).sort( function ( a, b ) {

				const distanceA = cache.objects.get( a ).distanceToCameraSquared;
				const distanceB = cache.objects.get( b ).distanceToCameraSquared;

				return distanceA - distanceB;

			} );

			const zMax = sorted.length;

			for ( let i = 0, l = sorted.length; i < l; i ++ ) {

				sorted[ i ].element.style.zIndex = zMax - i;

			}

		}

	}

}