import {
  Clock,
  PCFShadowMap,
  PerspectiveCamera,
  Scene,
  WebGLRenderer,
  sRGBEncoding
} from 'three'
import { EventDispatcher } from './EventDispatcher';
import { OrbitControls } from './OrbitControls/index';
import { RES } from './RES';

export enum EVENTS_ENUM {
  ENTERFRAME = 'ENTERFRAME',
}

// interface EventsCall {
//   func: (e: EventsCallArgsType) => any,
//   context: any
// }

// export interface EventsCallArgsType {
//   type: EventsType,
//   delta: number
// }

// type EventsMapInfer = {
//   [x in EventsType]?: EventsCall[]
// }

class ScenePreloadGroup extends EventDispatcher<EVENTS_ENUM> {
  get groupNames(): string[] { return null }

  constructor(...args: any[]) {
    super()
    
    this._preloadRes()
  }

  _preloadRes() {
    new Promise((resolve, reject) => {
      if (this.groupNames && this.groupNames.length) {
        Promise.all(
          this.groupNames.map(name => RES.loadGroup(name))
        ).then(resolve, reject)
      } else {
        resolve(1)
      }
    }).then(() => {
      this.ViewDraw()
    })
  }

  protected ViewDraw() {}
}

export class PerspectiveScene extends ScenePreloadGroup {
  scene: Scene
  camera: PerspectiveCamera
  clock: Clock
  renderer: WebGLRenderer
  rendererDom: HTMLCanvasElement
  OrbitControlsIns: OrbitControls

  renderCanvasElement: HTMLCanvasElement

  constructor(
    canvasDom?: HTMLCanvasElement
  ) {
    super()
    this.renderCanvasElement = canvasDom
    this.clock = new Clock()
    this.__initScene()
    this.__initCamera()
    this.__initRenderer()
    this.__initEvents()
  }

  /**
   * 初始化 场景
   */
  private __initScene() {
    const _scene = this.scene = new Scene()
  }

  private __initCamera() {
    const _camera = this.camera = new PerspectiveCamera(45, window.innerWidth / window.innerHeight, 1, 1000)
    _camera.position.set(10, 10, 10)
    _camera.lookAt(this.scene.position)
    this.scene.add(_camera)
  }

  private __initRenderer() {
    const matrialSceneCanvas = this.rendererDom = this.renderCanvasElement || document.createElement('canvas')
    matrialSceneCanvas.style.display = 'block'
    const _renderer = this.renderer = new WebGLRenderer({
      canvas: matrialSceneCanvas, // 如果指定了canvas，则不会创建一个新的canvas, 
      antialias: true, // 抗锯齿
    })
    // _renderer.shadowMap.enabled = true
    // _renderer.shadowMap.type = PCFShadowMap
    _renderer.setSize(window.innerWidth, window.innerHeight)
    _renderer.setPixelRatio(window.devicePixelRatio)
    _renderer.outputEncoding = sRGBEncoding;
    // document.body.appendChild(_renderer.domElement)

  }

  /**
   * 初始化事件
   */
  private __initEvents() {

    const OrbitControlsIns = this.OrbitControlsIns = new OrbitControls(this.camera, this.rendererDom)

    this.frameLoop()

    // window.addEventListener('resize', this.onWindowResize)
  }

  /**
   * 清除事件
   */
  private __removeEvents() {
    window.removeEventListener('resize', this.onWindowResize)
  }

  /**
   * window resize
   */
  private onWindowResize = () => {
    this.camera.aspect = window.innerWidth / window.innerHeight
    // 更新相机的投影矩阵
    this.camera.updateProjectionMatrix()
    this.renderer.setSize(window.innerWidth, window.innerHeight)
  }

  requestId: number
  /**
   * 帧循环
   */
  private frameLoop() {
    const delta = this.clock.getDelta()
    this.dispatchEvent(EVENTS_ENUM.ENTERFRAME, {
      delta,
    })

    // frame rander
    this.renderer.render(this.scene, this.camera)
    this.OrbitControlsIns.update()
    this.requestId = requestAnimationFrame(this.frameLoop.bind(this))
  }

  destroy() {
    cancelAnimationFrame(this.requestId)
    this.__removeEvents()
    this.removeAllEventListener()
  }
}