import {
  Vector3,
  Euler,
  EventDispatcher,
  Camera
} from 'three'

const _euler = new Euler(0, 0, 0, 'YXZ');
const _vector = new Vector3();
const _cameraDefaultVector = new Vector3(0, 0, -1);

const _changeEvent = { type: 'change' };
const _lockEvent = { type: 'lock' };
const _unlockEvent = { type: 'unlock' };

const _PI_2 = Math.PI / 2;

export class FirstPersonCameraControl extends EventDispatcher {
  domElement: HTMLElement
  isLocked: boolean // 是否锁定
  minPolarAngle: number // 最小相机俯仰角度
  maxPolarAngle: number // 最大相机俯仰角度
  camera: Camera // 相机
  lookSpeedCoefficient: number // 旋转速度系数

  constructor(camera: Camera, domElement?: HTMLElement) {
    super()

    if (!camera) {
      throw new Error('FirstPersonCameraControl: The first parameter "camera" is required')
    }

    if ( domElement === undefined ) {
			console.warn( 'FirstPersonCameraControl: The second parameter "domElement" is now mandatory.' );
			domElement = document.body;
		}

		this.domElement = domElement;
    this.camera = camera
		this.isLocked = false;

    // 设置相机俯仰角度 Range is 0 to Math.PI
		this.minPolarAngle = 0; // radians
		this.maxPolarAngle = Math.PI; // radians

    this.lookSpeedCoefficient = 0.001

    this.initEvents()
  }

  private initEvents() {
    const ownerDocument = this.domElement.ownerDocument

    ownerDocument.addEventListener('mousemove', this.onMouseMove, false)
    ownerDocument.addEventListener('touchmove', this.onMouseMove, false)
    ownerDocument.addEventListener('pointerlockchange', this.onPointerLockChange, false)
    ownerDocument.addEventListener('pointerlockerror', this.onPointerLockError, false)
  }

  private removeEvents() {
    const ownerDocument = this.domElement.ownerDocument
    ownerDocument.removeEventListener('mousemove', this.onMouseMove, false)
    ownerDocument.removeEventListener('touchmove', this.onMouseMove, false)
    ownerDocument.removeEventListener('pointerlockchange', this.onPointerLockChange, false)
    ownerDocument.removeEventListener('pointerlockerror', this.onPointerLockError, false)
  }

  private onPointerLockError() {
    console.error('FirstPersonCameraControl: Pointer lock error')
  }

  private onMouseMove = (event: Event) => {
    if (this.isLocked) {
      const camera = this.camera
      // @ts-ignore
      const movementX = event.movementX || event.mozMovementX || event.webkitMovementX || 0
      // @ts-ignore
      const movementY = event.movementY || event.mozMovementY || event.webkitMovementY || 0
      // 获取 相机的 四元 quaternion
      _euler.setFromQuaternion( camera.quaternion );
			_euler.y -= movementX * this.lookSpeedCoefficient;
			_euler.x -= movementY * this.lookSpeedCoefficient;
			_euler.x = Math.max( _PI_2 - this.maxPolarAngle, Math.min( _PI_2 - this.minPolarAngle, _euler.x ) );

			camera.quaternion.setFromEuler( _euler );

			this.dispatchEvent( _changeEvent );
    }
  }

  /**
   * pointerlock change 事件
   */
  private onPointerLockChange = () => {
    if ( this.domElement.ownerDocument.pointerLockElement === this.domElement ) {
      this.dispatchEvent( _lockEvent );
      this.isLocked = true;
    } else {
      this.dispatchEvent( _unlockEvent );
      this.isLocked = false;
    }
  }

  dispose() {
    this.removeEvents()
  }

  getCamera() {
    return this.camera
  }

  /**
   * 获取相机的四元朝向
   */
  getCameraDirection = (v: Vector3) => {
    // 简单粗暴点就下面这样 getWorldDirection
    return this.camera.getWorldDirection(v)
    return v.copy(_cameraDefaultVector).applyQuaternion(this.camera.quaternion)
  }

  /**
   * 向前移动
   * @param distance 
   */
  moveForward(distance: number) {
    const camera = this.camera
    // camera.getWorldDirection(v) 这样也行
    // 第一列为右向的向量
    // 第二 列为上向的向量
    // 第三 列为向前的向量
    _vector.setFromMatrixColumn( camera.matrix, 0 );
    // 向前的向量
    _vector.crossVectors( camera.up, _vector );
    
    camera.position.addScaledVector( _vector, distance );
  }

  getControlsRight() {
    const _camera = this.camera
    _vector.setFromMatrixColumn( _camera.matrix, 0 );
    return _vector
  }

  /**
   * 向右移动
   * @param distance 
   */
  moveRight(distance: number) {
    const _camera = this.camera
    _vector.setFromMatrixColumn( _camera.matrix, 0 );

    _camera.position.addScaledVector( _vector, distance );
  }

  moveToPostion(position: Vector3) {
    this.camera.position.copy(position)
  }

  add(v: Vector3) {
    this.camera.position.add(v)
  }

  /**
   * 锁定
   */
  lock() {
    this.domElement.requestPointerLock()
  }

  /**
   * 接触锁定
   */
  unlock() {
    this.domElement.ownerDocument.exitPointerLock()
  }
}