import { KeyBoardStateStore } from './../module/keyboardState/keyboardState';
import * as THREE from "three";
import { EventDispatcher } from "../module/EventDispatcher";
import { GLTFLoader } from '../module/loaders/GLTFLoader';
import { PreloadGroup } from '../module/UI/PreloadGroup/PreloadGroup';
import { RES } from '../module/RES';
import { GameConfig } from './GameConfig';

const materialDynamic = new THREE.MeshPhongMaterial( { color:0xfca400 } );
const materialStatic = new THREE.MeshPhongMaterial( { color:0x999999 } );
const materialInteractive = new THREE.MeshPhongMaterial( { color:0x990000 } );

var steeringIncrement = .04; // 转向速度
var steeringClamp = .5 //  Math.PI / 6; // .5
var maxEngineForce = 2000;
var maxBreakingForce = 100;


// Wheels
var FRONT_LEFT = 0;
var FRONT_RIGHT = 1;
var BACK_LEFT = 2;
var BACK_RIGHT = 3;

function createElement(type: string, className: string, style: Partial<CSSStyleDeclaration>, events?: {
  [eventName: string]: (event: Event) => void
}) {
  const dom = document.createElement(type)
  dom.setAttribute('class', className)
  
  for (let k in style) {
    dom.style[k] = style[k]
    // @ts-expect-error
    dom.originColor = style.backgroundColor
    // @ts-expect-error
    dom.touchColor = 'red'
  }
  document.body.appendChild(dom)

  for (let k in events) {
    dom.addEventListener(k, events[k])
  }

  return dom
}

type ActionNameEnum = 'acceleration' | 'braking' | 'left' | 'right';

const GLTFLoaderIns = new GLTFLoader()

interface GltfModel {
  animations: THREE.AnimationClip[],
  scene: THREE.Scene,
}

const VehicleDebugConfig = {
  debugger: false
}


export class Vehicle extends PreloadGroup{
  chassisMesh: THREE.Group // 车身
  fixedChassicMesh: THREE.Group // 固定的车身
  wheelMeshes: any[] = []
  chassisRidigBody: Ammo.btRigidBody
  raycastVehicle: Ammo.btRaycastVehicle
  btVehicleTuning: Ammo.btVehicleTuning

  luntaiIns: GltfModel // 轮胎模型

  speedNum: number

  constructor(pos: THREE.Vector3, quat: THREE.Quaternion) {
    super()

    // this.initChassisMesh()
    this.createVehicle(pos, quat)

    // this.addEventListener('update', this.update)
  }

  initChassisMesh() {
    this.add(new THREE.Mesh(new THREE.BoxGeometry(1,1,1), new THREE.MeshBasicMaterial({color: 0xffffff})))
  }

  createVehicle(pos: THREE.Vector3, quat: THREE.Quaternion) {
    // Vehicle contants
    // classis 底盘
    var chassisWidth = 1.8;
    var chassisHeight = 1;
    var chassisLength = 4;
    var massVehicle = 800;

    // Chassis
    var geometry = new Ammo.btBoxShape(new Ammo.btVector3(chassisWidth * .5, chassisHeight * .5, chassisLength * .5));
    // get btTransform
    var transform = new Ammo.btTransform();
    transform.setIdentity();
    transform.setOrigin(new Ammo.btVector3(pos.x, pos.y, pos.z));
    transform.setRotation(new Ammo.btQuaternion(quat.x, quat.y, quat.z, quat.w));

    var motionState = new Ammo.btDefaultMotionState(transform);
    var localInertia = new Ammo.btVector3(0, 0, 0);
    geometry.calculateLocalInertia(massVehicle, localInertia); // 计算局部惯性

    var body = this.chassisRidigBody = new Ammo.btRigidBody(new Ammo.btRigidBodyConstructionInfo(massVehicle, motionState, geometry, localInertia));
    body.setActivationState(4);

    var chassisMesh = this.chassisMesh = this.createChassisMesh(chassisWidth, chassisHeight, chassisLength);
    const gltf: GltfModel = RES.getRes('qiche_V6.gltf')
    chassisMesh.add(gltf.scene)
    gltf.scene.visible = !VehicleDebugConfig.debugger
    gltf.scene.translateY(-0.9)

    gltf.scene.traverse(function(obj) {
      // @ts-ignore
      if (obj.isMesh) {
        obj.castShadow = true
      }
    })
    // this.fixedChassicMesh = this.createChassisMesh(chassisWidth, chassisHeight, chassisLength)

    // this.fixedChassicMesh = this.chassisMesh.add(new THREE.Mesh(
    //   new THREE.BoxGeometry(1, 1, 1),
    //   materialDynamic
    // ))
    
  }

  /**
   * 添加轮子
   * @param isFront 是否是前轮
   * @param pos 坐标
   * @param radius 半径
   * @param width 宽度
   * @param index 索引值
   */
  addWheel(isFront: boolean, pos: Ammo.btVector3, radius: number, width: number, index: number) {
    const vehicle = this.raycastVehicle
    const tuning = this.btVehicleTuning
    var friction = 10000;

    // suspension 暂停 Stiffness刚度
    var suspensionStiffness = 20.0;
    var suspensionDamping = 2.3;
    var suspensionCompression = 4.4;
    var suspensionRestLength = 0.6;
    var rollInfluence = 0.2; // roll influence

    var wheelDirectionCS0 = new Ammo.btVector3(0, -1, 0);
    var wheelAxleCS = new Ammo.btVector3(-1, 0, 0);

    var wheelInfo = vehicle.addWheel(
        pos,
        wheelDirectionCS0,
        wheelAxleCS,
        suspensionRestLength,
        radius,
        tuning,
        isFront);

    // 悬架刚度？ set_m_suspensionStiffness  set_m_frictionSlip 有点冲突好像
    wheelInfo.set_m_suspensionStiffness(suspensionStiffness);
    wheelInfo.set_m_wheelsDampingRelaxation(suspensionDamping);
    wheelInfo.set_m_wheelsDampingCompression(suspensionCompression);
    wheelInfo.set_m_frictionSlip(friction); // 摩擦系数
    wheelInfo.set_m_rollInfluence(rollInfluence);


    this.wheelMeshes[index] = this.createWheelMesh(radius, width, pos);
  }

  createWheelMesh = (radius: number, width: number, pos: Ammo.btVector3) => {
    const group = new THREE.Group()
    var t = new THREE.CylinderGeometry(radius, radius, width, 24, 1);
    t.rotateZ(Math.PI / 2);
    var mesh = new THREE.Mesh(t, materialInteractive);
    mesh.add(new THREE.Mesh(new THREE.BoxGeometry(width * 1.5, radius * 1.75, radius*.25, 1, 1, 1), materialInteractive));
    mesh.castShadow = VehicleDebugConfig.debugger
    mesh.visible = VehicleDebugConfig.debugger
    group.add(mesh);
    const luntai = this.luntaiIns.scene.clone()
    luntai.visible = !VehicleDebugConfig.debugger
    group.add(luntai)
    this.add(group)
    group.position.set(pos.x(), pos.y(), pos.z())
    return group;
  }

  createChassisMesh(w: number, l: number, h: number) {
    const group = new THREE.Group()
    var shape = new THREE.BoxGeometry(w, l, h, 1, 1, 1);
    var mesh = new THREE.Mesh(shape, materialInteractive);
    mesh.castShadow = true
    mesh.visible = VehicleDebugConfig.debugger
    group.add(mesh);
    this.add(group)
    return group;
  }

  async initPhysics(physicsWorld: Ammo.btDiscreteDynamicsWorld) {
    physicsWorld.addRigidBody(this.chassisRidigBody)


    var tuning = this.btVehicleTuning = new Ammo.btVehicleTuning(); // 创建了一个 vehicleTurning 对象？
    var rayCaster = new Ammo.btDefaultVehicleRaycaster(physicsWorld); // 创建了一个射线投射

    var vehicle = this.raycastVehicle = new Ammo.btRaycastVehicle(tuning, this.chassisRidigBody, rayCaster); // 通过车辆和射线投射创建了一个车辆
    // 设置坐标系统，rightIndex为右方向的索引，upIndex为上方向索引，forwardIndex为前进方向索引
    vehicle.setCoordinateSystem(0, 1, 2); // 设置坐标系?
    physicsWorld.addAction(vehicle); // addAction

    var wheelAxisPositionBack = -1.3;
    var wheelRadiusBack = .4;
    var wheelWidthBack = .3;
    var wheelHalfTrackBack = 0.8;
    var wheelAxisHeightBack = 0;

    var wheelAxisFrontPosition = 1.5;
    var wheelHalfTrackFront = 0.8;
    var wheelAxisHeightFront = 0;
    var wheelRadiusFront = .35;
    var wheelWidthFront = 0.2;


    const LuntaiIns: GltfModel = this.luntaiIns = RES.getRes('luntai.gltf') 
    LuntaiIns.scene.traverse(function(obj) {
      // @ts-ignore
      if (obj.isMesh) {
        obj.castShadow = true
      }
    })
    this.addWheel(true, new Ammo.btVector3(wheelHalfTrackFront, wheelAxisHeightFront, wheelAxisFrontPosition), wheelRadiusFront, wheelWidthFront, FRONT_LEFT);
    this.addWheel(true, new Ammo.btVector3(-wheelHalfTrackFront, wheelAxisHeightFront, wheelAxisFrontPosition), wheelRadiusFront, wheelWidthFront, FRONT_RIGHT);
    this.addWheel(false, new Ammo.btVector3(-wheelHalfTrackBack, wheelAxisHeightBack, wheelAxisPositionBack), wheelRadiusBack, wheelWidthBack, BACK_LEFT);
    this.addWheel(false, new Ammo.btVector3(wheelHalfTrackBack, wheelAxisHeightBack, wheelAxisPositionBack), wheelRadiusBack, wheelWidthBack, BACK_RIGHT);
  }

  // Raycast Vehicle
  engineForce: number = 0;
  vehicleSteering: number = 0; // 车辆转向
  breakingForce: number = 0; // 刹车力

  actions: {
    [x in ActionNameEnum]: boolean
  } = {
    acceleration: false,
    braking: false,
    right: false,
    left: false
  }

  doAction = (actionName: ActionNameEnum) => {
    this.actions[actionName] = true
  }

  cancelAction = (actionName: ActionNameEnum) => {
    this.actions[actionName] = false
  }

  update(delta: number) {
    const vehicle = this.raycastVehicle
    const tuning = this.btVehicleTuning
    // vehicle.updateFriction(delta)

    // 记录当前速度
    const speed = this.speedNum = vehicle.getCurrentSpeedKmHour();

    let breakingForce = 0;
    let engineForce = 0;
    const isAccelerating = this.actions.acceleration || KeyBoardStateStore.isPressed('w')
    const isBreaking = this.actions.braking || KeyBoardStateStore.isPressed('s')
    const turnRight = this.actions.right || KeyBoardStateStore.isPressed('d')
    const turnLeft = this.actions.left || KeyBoardStateStore.isPressed('a')
    let stopFlag = true

    // if (!isAccelerating && !isBreaking && speed !== 0) {
    //   if (Math.abs(speed) < 0.2) {
    //     speed = 0
    //   } else {
    //     if (speed > 0) {
    //       breakingForce = maxBreakingForce / 4;
    //     } else {
    //       engineForce = maxEngineForce;
    //     }
    //   }
      
    // }

    // 最大速度
    if (speed < GameConfig.maxSpeed) {
      if (isAccelerating) {
        stopFlag = false
        if (speed < -1) {
          breakingForce = maxBreakingForce;
        } else {
          engineForce = maxEngineForce;
        }
      }
      if (isBreaking) {
        stopFlag = false
        if (speed > 1)
          breakingForce = maxBreakingForce;
        else engineForce = -maxEngineForce / 2;
      }
      if (turnLeft) {
        if (this.vehicleSteering < steeringClamp)
          this.vehicleSteering += steeringIncrement;
      }
      else {
        if (turnRight) {
          if (this.vehicleSteering > -steeringClamp)
            this.vehicleSteering -= steeringIncrement;
        }
        else {
          if (this.vehicleSteering < -steeringIncrement)
            this.vehicleSteering += steeringIncrement;
          else {
            if (this.vehicleSteering > steeringIncrement)
              this.vehicleSteering -= steeringIncrement;
            else {
              this.vehicleSteering = 0;
            }
          }
        }
      }
    }

    vehicle.applyEngineForce(engineForce, BACK_LEFT);
    vehicle.applyEngineForce(engineForce, BACK_RIGHT);
    // brake 刹车
    vehicle.setBrake(stopFlag ? 25 : breakingForce / 2, FRONT_LEFT);
    vehicle.setBrake(stopFlag ? 25 : breakingForce / 2, FRONT_RIGHT);
    vehicle.setBrake(breakingForce, BACK_LEFT);
    vehicle.setBrake(breakingForce, BACK_RIGHT);

    // 设置转向角度
    vehicle.setSteeringValue(this.vehicleSteering, FRONT_LEFT);
    vehicle.setSteeringValue(this.vehicleSteering, FRONT_RIGHT);

    var tm, p, q, i;
    var n = vehicle.getNumWheels();
    const wheelMeshes = this.wheelMeshes
    const chassisMesh = this.chassisMesh
    for (i = 0; i < n; i++) {
      vehicle.updateWheelTransform(i, true);
      // 更新 three 的visual wheels
      tm = vehicle.getWheelTransformWS(i);
      p = tm.getOrigin();
      q = tm.getRotation();
      wheelMeshes[i].position.set(p.x(), p.y(), p.z());
      wheelMeshes[i].quaternion.set(q.x(), q.y(), q.z(), q.w());
    }

    tm = vehicle.getChassisWorldTransform();
    p = tm.getOrigin();
    q = tm.getRotation();
    chassisMesh.position.set(p.x(), p.y(), p.z());
    chassisMesh.quaternion.set(q.x(), q.y(), q.z(), q.w());
  }  
}