import * as THREE from "three"


interface RigidBodyConfig {
  mass: number,
  friction: number // 摩擦力
  restitution: number // 弹性
  pos: THREE.Vector3
  qua: THREE.Quaternion
  scaleNum: number
}

/**
 * 创建物理
 * @param physicsShape 
 * @param cfg 
 * @returns 
 */
export function createRigidBody(physicsShape: Ammo.btCollisionShape, cfg: Partial<RigidBodyConfig> = {
  mass: 1
}) {
  const {
    mass,
    friction,
    restitution,
    pos,
    qua
  } = cfg
  const transfrom = new Ammo.btTransform()
  transfrom.setIdentity()
  pos && transfrom.setOrigin(new Ammo.btVector3(pos.x, pos.y, pos.z))
  qua && transfrom.setRotation(new Ammo.btQuaternion(qua.x, qua.y, qua.z, qua.w))
  const motionState = new Ammo.btDefaultMotionState(transfrom)

  const localInertia = new Ammo.btVector3(0,0,0)
  physicsShape.calculateLocalInertia(mass, localInertia)

  const rbInfo = new Ammo.btRigidBodyConstructionInfo(
    mass,
    motionState,
    physicsShape,
    localInertia
  )
  const rigidBody = new Ammo.btRigidBody(rbInfo)
  friction && rigidBody.setFriction(friction)
  restitution && rigidBody.setRestitution(restitution)

  if (mass > 0) {
    rigidBody.setActivationState(4)
  }
  // 自己加一个属性
  rigidBody._mass = mass

  return rigidBody
}

const defaultMatrix3 = new THREE.Matrix3()

export function createRigidBodyByThreeObject(obj3: THREE.Mesh, cfg: Partial<RigidBodyConfig>) {
  const {
    mass,
    friction,
    restitution,
    pos,
    qua,
    scaleNum = 1
  } = cfg
  // 反正每一帧都回去改， 所以无所谓的
  pos && obj3.position.copy(pos)
  qua && obj3.quaternion.copy(qua)
  if (scaleNum) {
    // obj3.scale.set(scaleNum, scaleNum, scaleNum)
    obj3.position.multiplyScalar(scaleNum)
  }
  defaultMatrix3.setFromMatrix4(obj3.matrixWorld)
  const body = createTriangleShapeByGeometry(obj3.geometry, {
    ...cfg,
    pos: obj3.position, //.applyMatrix3(defaultMatrix3), //(new THREE.Vector3()).applyMatrix4(obj3.matrixWorld),
    qua: obj3.quaternion
  })

  body.meshData = obj3
  obj3.geometry.userData.physicsBody = body

  return body
}


function tt(n: number, vertices: ArrayLike<number>, s: number) {
  const index = n * 3
  // console.log(n)
  // console.log(vertices[index] * s,
  //   vertices[index + 1] * s,
  //   vertices[index + 2] * s)
  return new Ammo.btVector3(
    vertices[index] * s,
    vertices[index + 1] * s,
    vertices[index + 2] * s
  )
}

export function createTriangleShapeByGeometry(geometry: THREE.BufferGeometry, cfg: Partial<RigidBodyConfig>) {
  const mesh = new Ammo.btTriangleMesh(true, true)
  const vertices = geometry.attributes.position.array
  const n = cfg.scaleNum || 1
  const indexArr = geometry.index.array
  for (let i = 0; i < indexArr.length; i += 3) {
    const trianglePoint = ([0, 1, 2] as const).map(num => {
      return tt(indexArr[i + num], vertices, n)
    })
    mesh.addTriangle(
      // @ts-ignore
      ...trianglePoint
    )
  }
  const shape = new Ammo.btBvhTriangleMeshShape(mesh, true, true)
  // ??
  shape.setMargin(0.05)
  return createRigidBody(shape, cfg)
}

export function createAmmoTerrainBody(positionArray: Float32Array, segmentArgs: number = 3, scale: number = 100) {
  const verties = positionArray
  const heightData: Record<string, Record<string, string>> = {}
  const segment = segmentArgs
  let terrainMinHeight = 0
  let terrainMaxHeight = 0
  let terrainMinX = 0
  let terrainMaxX = 0
  let terrainMinZ = 0
  let terrainMaxZ = 0
  for (let i = 0; i < verties.length; i+=3) {
    const x = verties[i] * scale
    const y = verties[i + 1] * scale
    const z = verties[i + 2] * scale
    const _x = Math.floor(x / segment)
    const _z = Math.floor(z / segment)
    // @ts-ignore
    const arr = (heightData[_z] || (heightData[_z] = {}))
    const pre = arr[_x] || '0/0/0';
    const spt = pre.split('/');
    const currSum = parseFloat(spt[0]) + y;
    const currCount = parseFloat(spt[1]) + 1;
    const currA = currSum / currCount
    terrainMinHeight = currA < terrainMinHeight ? currA : terrainMinHeight
    terrainMaxHeight = currA > terrainMaxHeight ? currA : terrainMaxHeight

    terrainMinX = x < terrainMinX ? x : terrainMinX
    terrainMaxX = x > terrainMaxX ? x : terrainMaxX

    terrainMinZ = z < terrainMinZ ? z : terrainMinZ
    terrainMaxZ = z > terrainMaxZ ? z : terrainMaxZ

    arr[_x] = `${currSum}/${currCount}/${currA}`
  }

  const terrainWidthExtents = terrainMaxX - terrainMinX
  const terrainDepthExtents = terrainMaxZ - terrainMinZ

  const terrainDepthOfKeys = Object.keys(heightData).sort((a, b) => Number(a) - Number(b))
  const terrainWidthOfKeys = Object.keys(heightData['0']).sort((a, b) => Number(a) - Number(b))

  const terrainWidth = terrainWidthOfKeys.length
  const terrainDepth = terrainDepthOfKeys.length

  const planMesh = new THREE.Mesh(
    new THREE.PlaneGeometry( terrainWidthExtents,terrainDepthExtents, terrainWidth - 1, terrainDepth -1),
    new THREE.MeshBasicMaterial({
      color: 0x000000,
      wireframe: true
    })
  )

  planMesh.rotateX( - Math.PI / 2 );

  const vertices = planMesh.geometry.attributes.position.array;
  console.log(planMesh.geometry.attributes.position)
  // i < l

  // let ii = 0;
  const v = Object.keys(heightData).sort((a, b) => Number(a) - Number(b));
  console.log(vertices)
  for ( var i  = 0; i < v.length; i ++) {
    const u = heightData[v[i]]
    const k = Object.keys(u).sort((a, b) => Number(a) - Number(b))
    for( let j = 0; j < k.length; j++) {
      const q = u[k[j]]
      const vv = +q.split('/').pop();
            // @ts-ignore 111

      vertices[(i * k.length + j) * 3 + 2]  = vv;

    }
  }
  planMesh.geometry.computeVertexNormals();

  console.log(heightData, terrainWidthExtents, terrainDepthExtents, terrainWidth, terrainDepth)

  console.log(terrainMaxX, terrainMinX)
  // This parameter is not really used, since we are using PHY_FLOAT height data type and hence it is ignored
  var heightScale = 1;

  // Up axis = 0 for X, 1 for Y, 2 for Z. Normally 1 = Y is used.
  var upAxis = 1;

  // hdt, height data type. "PHY_FLOAT" is used. Possible values are "PHY_FLOAT", "PHY_UCHAR", "PHY_SHORT"
  var hdt = "PHY_FLOAT"

  // "PHY_FLOAT";

  // Set this to your needs (inverts the triangles)
  var flipQuadEdges = false;

  // Creates height data buffer in Ammo heap
  const ammoHeightData = Ammo._malloc( 4 * terrainWidth * terrainDepth );
  console.log('ammoHeightData:', ammoHeightData)

  // Copy the javascript height data array to the Ammo one.
  var p = 0;
  var p2 = 0;

  for ( var j = 0; j < terrainDepthOfKeys.length; j ++ ) {
    for ( var i = 0; i < terrainWidthOfKeys.length; i ++ ) {
      const t = heightData[terrainDepthOfKeys[j]][terrainWidthOfKeys[i]] || '0/0/0'
      // write 32-bit float data to memory
      Ammo.HEAPF32[ammoHeightData + p2 >> 2] = +t.split('/').pop();

      p ++;

      // 4 bytes/float
      p2 += 4;
    }
  }

  // Creates the heightfield physics shape
  var heightFieldShape = new Ammo.btHeightfieldTerrainShape(

    terrainWidth,
    terrainDepth,

    ammoHeightData,

    heightScale,
    terrainMinHeight,
    terrainMaxHeight,

    upAxis,
    // @ts-expect-error
    hdt,
    flipQuadEdges
  );

  // Set horizontal scale
  var scaleX = terrainWidthExtents / ( terrainWidth - 1 );
  var scaleZ = terrainDepthExtents / ( terrainDepth - 1 );
  heightFieldShape.setLocalScaling( new Ammo.btVector3( scaleX, 1, scaleZ ) );

  heightFieldShape.setMargin( 0.05 );

  var groundTransform = new Ammo.btTransform();
  groundTransform.setIdentity();
  // Shifts the terrain, since bullet re-centers it on its bounding box.
  groundTransform.setOrigin( new Ammo.btVector3( 0, ( terrainMaxHeight + terrainMinHeight ) / 2, 0 ) );
  var groundMass = 0;
  var groundLocalInertia = new Ammo.btVector3( 0, 0, 0 );
  var groundMotionState = new Ammo.btDefaultMotionState( groundTransform );
  var groundBody = new Ammo.btRigidBody( new Ammo.btRigidBodyConstructionInfo( groundMass, groundMotionState, heightFieldShape, groundLocalInertia ) );
  return [groundBody, planMesh]
}