import { EVENTS_ENUM, PerspectiveScene } from "../module/PerspectiveScene";
import * as THREE from 'three'
import { OimoPhysicWorld } from "../OimoPhysicWorld/OimoPhysicWorld";
import { AmmoPhysicWorld } from "../module/AmmoPhysicWorld";
import { Vehicle } from "./vehicle";
import { FirstPersonCameraControl } from "../module/CameraControl/FirstControlCameraControl";
import { GLTFLoader } from "../module/loaders/GLTFLoader";
import { createAmmoTerrainBody, createRigidBodyByThreeObject, createTriangleShapeByGeometry } from "./tools";
import { MeshBasicMaterial } from "three";
import Stats from 'three/examples/jsm/libs/stats.module'
import { RoomEnvironment } from '../module/environment/RoomEnvironment';
import { RES } from "../module/RES";
import { GameConfig } from "./GameConfig";
import { Dom } from "./dom";
import { ResJson } from "../ResJson";
import { GDispather } from "../message";

function gradTexture(color: [number[], string[]]) {
  var c = document.createElement("canvas");
  var ct = c.getContext("2d");
  c.width = 16; c.height = 256;
  var gradient = ct.createLinearGradient(0,0,0,256);
  var i = color[0].length;
  while(i--){ gradient.addColorStop(color[0][i],color[1][i]); }
  ct.fillStyle = gradient;
  ct.fillRect(0,0,16,256);
  var texture = new THREE.Texture(c);
  texture.needsUpdate = true;
  return texture;
}

function basicTexture(n: number){
  var canvas = document.createElement( 'canvas' );
  canvas.width = canvas.height = 64;
  var ctx = canvas.getContext( '2d' );
  var colors = [];
  if(n===0){ // sphere
      colors[0] = "#58AA80";
      colors[1] = "#58FFAA";
  }
  if(n===1){ // sphere sleep
      colors[0] = "#383838";
      colors[1] = "#38AA80";
  }
  if(n===2){ // box
      colors[0] = "#AA8058";
      colors[1] = "#FFAA58";
  }
  if(n===3){ // box sleep
      colors[0] = "#383838";
      colors[1] = "#AA8038";
  }
  ctx.fillStyle = colors[0];
  ctx.fillRect(0, 0, 64, 64);
  ctx.fillStyle = colors[1];
  ctx.fillRect(0, 0, 32, 32);
  ctx.fillRect(32, 32, 32, 32);

  var tx = new THREE.Texture(canvas);
  tx.needsUpdate = true;
  return tx;
}

const GeometryMap = {
  'box': new THREE.BoxGeometry(1,1,1)
}


function setStyle(dom: HTMLElement, styles: Partial<CSSStyleDeclaration>) {
  for (let k in styles) {
    dom.style[k] = styles[k]
  }
}

var ZERO_QUATERNION = new THREE.Quaternion(0, 0, 0, 1);

const GLTFLoaderIns = new GLTFLoader()
const _defaultVector3 = new THREE.Vector3()
const _rayCaster = new THREE.Raycaster()
const _v3 = new THREE.Vector3()

export class CarScene extends PerspectiveScene {

  vehicleIns: Vehicle;
  cameraControls: FirstPersonCameraControl
  dirLight: THREE.DirectionalLight
  terrainMesh: THREE.Mesh
  cameraPos: THREE.Vector3

  get groupNames(): string[] {
    return ['gltf', 'skybox']
  }

  constructor(d: HTMLCanvasElement, u: HTMLElement) {
    super(d)
    
    this.camera.position.set(20, 20, 20)
    this.camera.far = 400
    this.OrbitControlsIns.enabled = false
    // this.OrbitControlsIns.enablePan = false
    // this.OrbitControlsIns.enableDamping = true

    // this.cameraControls = new FirstPersonCameraControl(this.camera)
    // document.body.addEventListener('click', () => {
    //   this.cameraControls.lock()
    // })
    this.cameraPos = GameConfig.CameraPosConfig.change()

    this.initStats()

    this.initEvents()
    this.init2DUi(u)

    this.initRenderer()
  }

  // 初始化 render
  initRenderer() {

    this.onDeviceOrientationChange()
  }

  initStats() {
    const statsIns = Stats()
    console.log('===========Stats: ============\n', statsIns)
    // 设置监视器面板，传入面板id（0: fps, 1: ms, 2: mb）
    statsIns.setMode(0)
    statsIns.domElement.style.position = 'fixed'
    statsIns.domElement.style.left = '0px'
    statsIns.domElement.style.top = '0px'
    document.body.appendChild(statsIns.domElement)

    this.addEventListener(EVENTS_ENUM.ENTERFRAME, function() {
      statsIns.update()
    })
  }

  ViewDraw() {
    const reflectionCube = new THREE.CubeTexture()
    const map: Record<string, string> = {
      0: 'px1.jpg',
      1: 'nx1.jpg',
      2: 'py1.jpg',
      3: 'ny1.jpg',
      4: 'pz1.jpg',
      5: 'nz1.jpg'
    }
    for (let k in map) {
      reflectionCube.image[k] = RES.getRes(map[(k)]).image
    }
    reflectionCube.encoding = THREE.sRGBEncoding;
    reflectionCube.needsUpdate = true
    this.scene.background = reflectionCube;

    // let pmremGenerator = new THREE.PMREMGenerator(this.renderer)
    // pmremGenerator.compileEquirectangularShader();
    // this.scene.environment = pmremGenerator.fromScene(new RoomEnvironment(), 0.04).texture;

    // this.scene.background = new THREE.Color(0x000a6b)
    // 下面的 fog 雾化 会影响最后成像
    this.scene.fog = new THREE.Fog(0xa0a0a0, 10, 400)

    const dirLight = this.dirLight = new THREE.DirectionalLight(0xffffff, 1)
    dirLight.castShadow = true
    dirLight.position.set(20, 20, 20)
    this.scene.add(dirLight)

    this.scene.add(new THREE.AmbientLight(0x222222, 1))

    dirLight.castShadow = true
    console.log(dirLight.shadow)
    dirLight.shadow.mapSize.width = 1024 // 2的 n次幂 数值越高 细节越高
    dirLight.shadow.mapSize.height = 1024
    dirLight.shadow.camera.near = 0.3
    dirLight.shadow.camera.far = 60
    dirLight.shadow.camera.top = 20
    dirLight.shadow.camera.left = 20
    dirLight.shadow.camera.bottom = -20
    dirLight.shadow.camera.right = -20
    dirLight.shadow.radius = 4

    let renderer = this.renderer
    // 开启阴影
    renderer.shadowMap.enabled = true
    // THREE.BasicShadowMap-性能非常好但是质量很差
    // THREE.PCFShadowMap-性能较差但边缘更平滑(默认)
    // THREE.PCFSoftShadowMap-性能较差但边缘更柔和  radius 不会生效
    // THREE.VSMShadowMap-性能差，约束多，但能够产生意想不到的效果。
    renderer.shadowMap.type = THREE.PCFSoftShadowMap

    this.scene.add(new THREE.AmbientLight(0x3D4143))

    if (GameConfig.helperDebuger) {
      this.scene.add(new THREE.DirectionalLightHelper(dirLight))
      this.scene.add(new THREE.CameraHelper(dirLight.shadow.camera))

      this.scene.add(new THREE.AxesHelper(100))
      this.scene.add(new THREE.GridHelper(100, 20))
    }

    // let imgTexture = new THREE.TextureLoader().load('judgeLineMap.png')
    // // @ts-ignore
    // window['imgTexture'] = imgTexture
    // console.log('=======imgTexture=======', imgTexture, imgTexture.image)


    this.initAmmoPhySicWorld()
  }

  UICtn: HTMLElement
  init2DUi(d: HTMLElement) {
    // @ts-ignore
    const remScale = 0.5 // window['remScale']  这里就写死吧
    this.UICtn = Dom.div(
      {
        attrs: {
          class: '2d-ui'
        },
        styles: {
          position: 'fixed',
          bottom: '20px',
          display: 'flex',
          justifyContent: 'space-between',
          alignItems: 'flex-end',
          width: '100%',
          boxSizing: 'border-box',
          padding: `0 ${100 * remScale}px`
        }
      },

      GameConfig.helperDebuger && Dom.div(
        {
          attrs: {
            class: 'tips'
          },
          styles: {
            position: 'fixed',
            right: '0',
            bottom: '0',
            padding: '3px 10px',
            backgroundColor: 'yellow'
          }
        },
        '暂未提交分数'
      )
      ,

      GameConfig.helperDebuger && Dom.div(
        {
          attrs: {
            id: 'speedBoard'
          },
          styles: {
            position: 'fixed',
            backgroundColor: 'yellow',
            padding: '3px 20px',
            left: "0px",
            bottom: "0px"
          }
        }
      ),

      Dom.div(
        {
          attrs: {
            class: 'camera-icon'
          },
          styles: {
            position: 'absolute',
            backgroundImage: `url(${ResJson.path}UI/camera-icon.png)`,
            backgroundSize: '100% 100%',
            width: `${67 * remScale}px`,
            height: `${67 * remScale}px`,
            right: `${100 * remScale}px`,
            top: `-${200 * remScale}px`
          },
          events: {
            touchstart: {
              callback: () => {
                this.cameraPos = GameConfig.CameraPosConfig.change()
              }
            }
          }
        }
      ),
      
      Dom.div(
        {
          attrs: {
            class: 'left-cont',
          },
          styles: {
            display: 'flex',
            alignItems: 'flex-end'
          }
        },
        Dom.div({
          attrs: {
            class: 'reset-icon'
          },
          styles: {
            backgroundImage: `url(${ResJson.path}UI/reset-icon.png)`,
            backgroundSize: '100% 100%',
            width: `${67 * remScale}px`,
            height: `${67 * remScale}px`
          },
          events: {
            click: {
              callback: () => {
                const pos = GameConfig.initVehiclePos
                const transformIns = this.vehicleIns.chassisRidigBody.getWorldTransform()
                transformIns.setOrigin(new Ammo.btVector3(pos.x, pos.y, pos.z))
                transformIns.setRotation(
                  new Ammo.btQuaternion(ZERO_QUATERNION.x, ZERO_QUATERNION.y, ZERO_QUATERNION.z, ZERO_QUATERNION.w)
                )
              }
            }
          }
        }),
  
        Dom.div({
          attrs: {
            class: 'left-icon',
          },
          styles: {
            backgroundImage: `url(${ResJson.path}UI/left-icon.png)`,
            backgroundSize: 'contain',
            backgroundPosition: 'center',
            backgroundRepeat: 'no-repeat',
            width: `${70 * remScale}px`,
            height: `${126 * remScale}px`,
            marginLeft: '60px',
            padding: '0 20px'
          },
          events: {
            touchstart: {
              callback: () => {
                this.vehicleIns.doAction('left')
              }
            },
            touchend: {
              callback: () => {
                this.vehicleIns.cancelAction('left')
              }
            }
          }
        }),

        Dom.div({
          attrs: {
            class: 'right-icon',
          },
          styles: {
            backgroundImage: `url(${ResJson.path}UI/right-icon.png)`,
            backgroundSize: 'contain',
            backgroundPosition: 'center',
            backgroundRepeat: 'no-repeat',
            width: `${70 * remScale}px`,
            height: `${126 * remScale}px`,
            marginLeft: '40px',
            padding: '0 20px'
          },
          events: {
            touchstart: {
              callback: () => {
                this.vehicleIns.doAction('right')
              }
            },
            touchend: {
              callback: () => {
                this.vehicleIns.cancelAction('right')
              }
            }
          }
        })
      ),

      Dom.div(
        {
          attrs: {
            class: 'right-cont'
          },
          styles: {
            display: 'flex',
            alignItems: 'flex-end'
          }
        },

        Dom.div({
          attrs: {
            class: 'brake-icon',
          },
          styles: {
            backgroundImage: `url(${ResJson.path}UI/brake-icon.png)`,
            backgroundSize: 'contain',
            backgroundPosition: 'center',
            backgroundRepeat: 'no-repeat',
            width: `${102 * remScale}px`,
            height: `${150 * remScale}px`,
          },
          events: {
            touchstart: {
              callback: () => {
                this.vehicleIns.doAction('braking')
              }
            },
            touchend: {
              callback: () => {
                this.vehicleIns.cancelAction('braking')
              }
            }
          }
        }),
  
        Dom.div({
          attrs: {
            class: 'engine-icon',
          },
          styles: {
            backgroundImage: `url(${ResJson.path}UI/engine-icon.png)`,
            backgroundSize: '100% 100%',
            width: `${93 * remScale}px`,
            height: `${185 * remScale}px`,
            marginLeft: '40px'
          },
          events: {
            touchstart: {
              callback: () => {
                this.vehicleIns.doAction('acceleration')
              }
            },
            touchend: {
              callback: () => {
                this.vehicleIns.cancelAction('acceleration')
              }
            }
          }
        })
      )
    )

    const uiNode = d || this.rendererDom.parentNode
    uiNode.appendChild(this.UICtn)
  }

  addMapWall() {
    const wallList: {
      rotationDeg: number,
      pos: THREE.Vector3
    }[] = [
      {
        rotationDeg: 0,
        pos: new THREE.Vector3(0, 10, 80)
      },
      {
        rotationDeg: Math.PI / 2,
        pos: new THREE.Vector3(-80, 10, 0)
      },
      {
        rotationDeg: 0,
        pos: new THREE.Vector3(0, 10, -80)
      },
      {
        rotationDeg: Math.PI / 2,
        pos: new THREE.Vector3(80, 10, 0)
      }
    ]

    for (let config of wallList) {
      const wall = new THREE.Mesh(
        new THREE.BoxGeometry(196, 20),
        new THREE.MeshBasicMaterial({
          wireframe: false,
          opacity: 0,
        })
      )
      wall.visible = false
      wall.quaternion.copy(
        new THREE.Quaternion().setFromAxisAngle(new THREE.Vector3(0, 1, 0), config.rotationDeg)
      )
      wall.position.copy(config.pos)
      this.scene.add(
        wall
      )
      this.ammoPhysicWorld.addMesh(wall)
    }

  }

  ammoPhysicWorld: AmmoPhysicWorld
  initAmmoPhySicWorld() {
    this.ammoPhysicWorld = new AmmoPhysicWorld()
    this.addMapWall()
    
    interface GltfModel {
      animations: THREE.AnimationClip[],
      scene: THREE.Scene,
    }

    const ins: GltfModel = RES.getRes('shamo_V14_L(1).gltf')
    
    let terrainMesh: THREE.Mesh
    this.scene.add(ins.scene)
    ins.scene.scale.set(200, 200, 200)
    ins.scene.traverse((obj: THREE.Object3D) => {
      // @ts-ignore
      if (obj.name.startsWith('qigan') && obj.isMesh) {
        obj.castShadow = true
        this.ammoPhysicWorld.addRigidBodyWithMeshObj(createRigidBodyByThreeObject(
          obj as THREE.Mesh,
          {
            mass: 0,
            scaleNum: 200
          }
        ))
      } else if (obj.name === 'shamo' && (obj as THREE.Mesh).isMesh) {
        terrainMesh = obj as THREE.Mesh
      } else if (obj.name.startsWith('shitou')) {
        obj.castShadow = true
        obj.receiveShadow = true
        this.ammoPhysicWorld.addRigidBodyWithMeshObj(createRigidBodyByThreeObject(
          obj as THREE.Mesh,
          {
            mass: 0,
            scaleNum: 200
          }
        ))
      } else if (obj.name.startsWith('qimian')) {
        obj.castShadow = true
      } else if (obj.name.startsWith('cao')) {
        // obj.castShadow = true
      }
    })
    this.terrainMesh = terrainMesh
    if (terrainMesh) {
      terrainMesh.receiveShadow = true
      // terrainMesh.castShadow = true
      // @ts-ignore
      terrainMesh.material.side = THREE.FrontSide
      // @ts-ignore
      // terrainMesh.material.wireframe = true
    }

    // 旗子动画
    const animationMixer = new THREE.AnimationMixer(ins.scene)
    const animationAction = animationMixer.clipAction(ins.animations[0])
    animationAction.play()
    this.addEventListener(EVENTS_ENUM.ENTERFRAME, (e) => {
      animationMixer.update(e.data.delta)
    })

    // // @ts-ignore
    // const [worldbody, mesh] = createAmmoTerrainBody(array, 6, 200)
    // // @ts-ignore
    // this.ammoPhysicWorld.physicWorld.addRigidBody(worldbody)
    // this.scene.add(mesh as THREE.Mesh)

    // 添加 地形
    this.ammoPhysicWorld.addRigidBodyWithMeshObj(createRigidBodyByThreeObject(
      terrainMesh,
      {
        scaleNum: 200,
        mass: 0
      }
    ))

    // 添加车
    const vehicleIns = this.vehicleIns = new Vehicle(GameConfig.initVehiclePos, ZERO_QUATERNION)
    this.scene.add(vehicleIns)
    this.ammoPhysicWorld.addDynamicBody(vehicleIns)
    // @ts-ignore
    window['vehicleIns'] = vehicleIns
    console.log(this.dirLight)
    this.dirLight && (this.dirLight.target = vehicleIns.chassisMesh)

    // const planeGeometry = new THREE.BoxGeometry(100, 2, 100)
    // // planeGeometry.rotateX( - Math.PI / 2 ); // 这里加了旋转矩阵 物理世界 记得要转换一下
    // const planeMesh = new THREE.Mesh(planeGeometry, new THREE.MeshPhongMaterial({
    //   color: 0xffffff,
    //   side: THREE.DoubleSide,
    //   // wireframe: true
    // }))
    // planeMesh.position.y = -1
    // // @ts-ignore
    // planeMesh.material = terrainMesh.material
    // this.scene.add(planeMesh)


    // this.ammoPhysicWorld.addMesh(planeMesh, {
    //   friction: 10,
    //   restitution: 0.1,
    //   mass: 0
    // })
    // this.ammoPhysicWorld.addRigidBodyWithMeshObj(createRigidBodyByThreeObject(
    //   planeMesh,
    //   {
    //     mass: 0
    //   }
    // ))

    if (GameConfig.collisionDebuger) {
      // 下面是测试模型数据
      const boxGeometry = new THREE.BoxGeometry(1,1,1, 10, 10, 10)
      const material = new THREE.MeshLambertMaterial({
        // wireframe: true,
        // map: new THREE.TextureLoader().load('http://qnpic.top/yoona2.jpg'),
        color: new THREE.Color(0xffffff * Math.random())
      })
      const boxMesh = new THREE.Mesh(boxGeometry, material)
      boxMesh.castShadow = true
      const rigidBody = createRigidBodyByThreeObject(boxMesh, {
        pos: new THREE.Vector3(5, 0.5, 5),
        mass: 0
      })
      this.scene.add(boxMesh)
      this.ammoPhysicWorld.addRigidBodyWithMeshObj(rigidBody)
      console.log('boxMesh: ', boxMesh)

      
      const path = new THREE.CatmullRomCurve3([
        new THREE.Vector3(0, 0, 0),
        new THREE.Vector3(0, 5, 0),
        new THREE.Vector3(5, 10, 5),
        new THREE.Vector3(10, 5, 10),
        new THREE.Vector3(10, 0, 10)
      ])

      const tt = new THREE.Mesh(
        new THREE.TubeGeometry(path, 40, 2, 8, false),
        new THREE.MeshPhongMaterial( { color:0x990000, wireframe: true } )
      )
      this.scene.add(tt)
      tt.position.y = 0
      // @ts-ignore
      window.tt = tt
      this.ammoPhysicWorld.addRigidBodyWithMeshObj(createRigidBodyByThreeObject(
        tt,
        {
          mass: 0,
          pos: new THREE.Vector3(0, 0, 5)
        }
      ))

      Array.from({
        length: 10
      }, (_) => {
        const boxGeometry = new THREE.BoxGeometry(1,1,1)
        const material = new THREE.MeshLambertMaterial({
          // wireframe: true,
          // map: new THREE.TextureLoader().load('http://qnpic.top/yoona2.jpg'),
          color: 0x000000
        })
        material.color = new THREE.Color(0xffffff * Math.random())
        const boxMesh = new THREE.Mesh(boxGeometry, material)
        boxMesh.position.set(-Math.random() * 20, 10, -Math.random() * 20)
        boxMesh.castShadow = true
        this.scene.add(boxMesh)
  
        this.ammoPhysicWorld.addMesh(boxMesh, {
          mass: 1,
          // friction: 10,
          restitution: 1
        })
      })
    }

    // Array.from({
    //   length: 10
    // }, (_, index) => {
    //   const boxGeometry = new THREE.BoxGeometry(1,1,1, 10, 10, 10)
    //   const material = new THREE.MeshLambertMaterial({
    //     // wireframe: true,
    //     // map: new THREE.TextureLoader().load('http://qnpic.top/yoona2.jpg'),
    //     color: new THREE.Color(0xffffff * Math.random())
    //   })
    //   const boxMesh = new THREE.Mesh(boxGeometry, material)
    //   boxMesh.castShadow = true
    //   const rigidBody = createRigidBodyByThreeObject(boxMesh, {
    //     pos: new THREE.Vector3(Math.random() * 10, 10, -Math.random() * 20),
    //     mass: 1,
    //     restitution: 1
    //   })
    //   this.scene.add(boxMesh)
    //   this.ammoPhysicWorld.addRigidBodyWithMeshObj(rigidBody)
    // })

    // const t1 = new THREE.BufferGeometry()
    // const vertices = new Float32Array([
    //   -15, 0, 15,
    //   15, 0, 15,
    //   15, 0, -15,
    //   -15, 0, -15
    // ])
    // t1.attributes.position = new THREE.BufferAttribute(vertices, 3)
    // t1.index = new THREE.BufferAttribute(
    //   new Uint16Array([
    //     0, 1, 2,
    //     0, 2, 3
    //   ]),
    //   1
    // )
    // const t1Mesh = new THREE.Mesh(t1, new THREE.MeshPhongMaterial({
    //   color: 0xffffff,
    //   side: THREE.DoubleSide,
    //   wireframe: true
    // }))
    // this.scene.add(t1Mesh)
    // this.ammoPhysicWorld.addRigidBodyWithMeshObj(createRigidBodyByThreeObject(
    //   t1Mesh,
    //   {
    //     mass: 0
    //   }
    // ))

    // this.scene.add(new THREE.Mesh(new THREE.PlaneBufferGeometry(100, 100, 99, 99), new THREE.MeshBasicMaterial({
    //   wireframe: true
    // }))).position.set(0, 4, 0)
  }

  physicWorld: OimoPhysicWorld
  /**
   * 创建物理世界
   */
  initOimoPhysicWorld() {
    this.physicWorld = new OimoPhysicWorld()

    console.log(this.physicWorld)
    const boxGeometry = new THREE.BoxGeometry(1,1,1)
    const material = new THREE.MeshLambertMaterial({
      wireframe: true
    })
    material.color = new THREE.Color(0xffffff * Math.random())
    const boxMesh = new THREE.Mesh(boxGeometry, material)
    boxMesh.position.set(0, 20, 0)
    this.scene.add(boxMesh)
    const _ =  this.physicWorld.addMesh(boxMesh, {
      massPos: [1,1,1],
      density: 1,
      friction: 1,
      // mass: 10000
    })

    const planeGeometry = new THREE.BoxGeometry(10, 2, 10)
    const planeMesh = new THREE.Mesh(planeGeometry, new THREE.MeshPhongMaterial({
      color: 0xffffff,
      side: THREE.DoubleSide
    }))
    planeMesh.receiveShadow = true
    // planeMesh.position.y = -2
    this.scene.add(planeMesh)

    const ridigBody = this.physicWorld.addMesh(planeMesh, {
      move: false
    })

    // @ts-expect-error
    window['test'] = function() {
      _.applyImpulse(new THREE.Vector3(0, 0, 0), new THREE.Vector3(1, 0,0))
      console.log(_)
    }
  }

  initEvents() {
    this.addEventListener(EVENTS_ENUM.ENTERFRAME, this.onEnterFrame, this)

    window.addEventListener("onorientationchange" in window ? "orientationchange" : "resize", this.onDeviceOrientationChange, false);
  }

  removeEvents() {
    this.removeEventListener(EVENTS_ENUM.ENTERFRAME, this.onEnterFrame)
    window.removeEventListener("onorientationchange" in window ? "orientationchange" : "resize", this.onDeviceOrientationChange, false)
  }

  destroy() {
    super.destroy()
    this.removeEvents()
    this.UICtn.remove()
  }

  
  timeoutId: any
  onDeviceOrientationChange = () => {
    this.timeoutId && clearTimeout(this.timeoutId)
    this.timeoutId = setTimeout(() => {
      let isPortrait = false // 是否是 竖屏
      if (window.orientation === 180 || window.orientation === 0) {
        isPortrait = true
        // 竖屏
        this.camera.aspect = window.innerHeight / window.innerWidth
        // 更新相机的投影矩阵
        this.camera.updateProjectionMatrix()
        this.renderer.setSize(window.innerHeight, window.innerWidth)
  
        setStyle(this.rendererDom.parentElement, {
          transform: 'rotate(90deg)',
          transformOrigin: 'calc(100vw / 2) calc(100vw / 2)'
        })

        this.UICtn.style.width = '100vh'
      }
      if (window.orientation === 90 || window.orientation === -90 ){
        isPortrait = false
        // 横屏
        this.camera.aspect = window.innerWidth / window.innerHeight
        // 更新相机的投影矩阵
        this.camera.updateProjectionMatrix()
        this.renderer.setSize(window.innerWidth, window.innerHeight)
  
        setStyle(this.rendererDom.parentElement, {
          transform: 'none',
          transformOrigin: 'calc(100vw / 2) calc(100vw / 2)'
        })

        this.UICtn.style.width = '100vw'
      }
      GDispather.dispatchEvent('orientationChange', {
        isPortrait
      })
    }, 50)
  }

  preSpeedNum: number = 0 // 前一帧速度
  preVehiclePositionY: number = 0 // 前一帧高度
  _currDistance: number = 0
  _preVehiclePos: THREE.Vector3 = new THREE.Vector3()
  _rotateNum: number = 0
  _prerotateflag: number = 0

  get currDistance() {
    return this._currDistance
  }

  set currDistance(v: number) {
    this._currDistance = v
    GDispather.dispatchEvent('distanceUpdateSync', {
      distance: v,
      circleNo: this.rotateNum.toFixed(2)
    })
  }

  get rotateNum() {
    return this._rotateNum
  }

  set rotateNum(v) {
    this._rotateNum = v
  }

  onEnterFrame(e: any) {
    const delta = e.data.delta
    if (this.physicWorld) {
      this.physicWorld.step()
    }

    if (this.ammoPhysicWorld) {
      this.ammoPhysicWorld.step()
    }
    
    this.updateCamera(this.camera)

    if (this.vehicleIns && this.UICtn) {
      const vehiclePos = this.vehicleIns.chassisMesh.position
      const h = vehiclePos.y
      const currSpeedNum = this.vehicleIns.speedNum
      const minH = GameConfig.h

      this.UICtn.querySelector('#speedBoard') && (this.UICtn.querySelector('#speedBoard').innerHTML = (currSpeedNum < 0 ? '(R) ' : '') + Math.abs(currSpeedNum).toFixed(1) + ' km/h');

      // 高于 一定高度
      if (h > minH) {
        const _t = new THREE.Vector3(vehiclePos.x, 0, vehiclePos.z)
        const currflag = _defaultVector3.crossVectors(this._preVehiclePos, _t).y
        let rn = 0

        if (this.preVehiclePositionY < minH) {
          console.log('开始积分')
          // 刚开始进入
          this.currDistance = 0
          this.rotateNum = 0
        } else {
          const isReversalSpeed = currSpeedNum * this.preSpeedNum < 0
          // 跑的方向 相反
          const isReversal = this._prerotateflag * currflag < 0
          rn = this._preVehiclePos.angleTo(_t)
          if (isReversalSpeed || isReversal) {
            this.onGameSubmit(this.currDistance)
            console.log('开始积分')
            this.currDistance = 0
            this.rotateNum = 0
          }
        }
        this.currDistance += currSpeedNum * delta
        this.rotateNum += rn
        
        this._preVehiclePos.copy(_t)
        this._prerotateflag = currflag
      } else {
        if (this.currDistance != 0) {
          // 提交
          this.onGameSubmit(this.currDistance)
          this.currDistance = 0
          this.rotateNum = 0
        }
      }

      this.preVehiclePositionY = h
      this.preSpeedNum = currSpeedNum
    }
    
  }

  onGameSubmit(n: number) {
    console.log('当前分数：', Math.abs(n), '圈数:', (this.rotateNum / Math.PI / 2).toFixed(2))
    GDispather.dispatchEvent('distanceUpdate', {
      distance: Math.abs(n),
      circleNo: this.rotateNum.toFixed(2)
    })
    if (this.UICtn.querySelector('.tips')) {
      this.UICtn.querySelector('.tips').textContent = '当前提交分数：' + Math.abs(n).toFixed(2)
    }
    
  }

  updateCamera(camera: THREE.Camera) {
    if (this.vehicleIns && !this.OrbitControlsIns.enabled) {
      const carPosition = this.vehicleIns.chassisMesh.position
      // 跟随 Car
      // 0, 8, -20
      // 0, 20, -16
      _defaultVector3.copy(this.cameraPos)
      // @ts-ignore
      window['vehicle'] || (window['vehicle'] = this.vehicleIns.chassisMesh)
      // const m3 = new THREE.Matrix3()
      // relativeCameraOffset.applyMatrix3(m3.setFromMatrix4(this.vehicleIns.chassisMesh.matrixWorld))
      const vehiclePosition = this.vehicleIns.chassisMesh.position
      _defaultVector3.applyMatrix4(this.vehicleIns.chassisMesh.matrixWorld)
      _defaultVector3.y = this.vehicleIns.chassisMesh.position.y + this.cameraPos.y
      _v3.subVectors(_defaultVector3, vehiclePosition)
      _rayCaster.near = 0
      _rayCaster.far = _v3.length()
      _rayCaster.set(vehiclePosition, _v3.normalize())
      const result = _rayCaster.intersectObject(this.terrainMesh)
      if (result.length) {
        const targetPoint = result[0].point.addScalar(0.9)
        // .addScalar(0.9)
        // console.log(targetPoint)
        _defaultVector3.copy(targetPoint)
      }
      this.camera.position.lerp(_defaultVector3, 0.1)
      // this.camera.position.copy(_defaultVector3)
      this.camera.lookAt(carPosition)

      
    }
    if (this.vehicleIns && this.dirLight) {
      const carPosition = this.vehicleIns.chassisMesh.position
      _defaultVector3.set(10, 10, 10)
      _defaultVector3.addVectors(_defaultVector3, carPosition)
      this.dirLight.position.copy(_defaultVector3)
      this.dirLight.lookAt(carPosition)
    }
    
  }
}