import { ParkourGameEvents, ParkourGameEleType, ConveyorBeltType, ParkourGameConfig, ConveyorItemConstrutType, UniqueKeyType } from './../ParkourGameConfig';
import { GDispatcher } from "../../Game"
import Clock from "../../tools/Clock"
import UI from '../../tools/UI';
import { RES } from '../../../module/RES';
import { GPool } from '../../../module/tools/GPool';
import { RectPhysicsCont, WithCollisionDetection } from '../CollisionComposite';

type PosPoint = {
  x: number,
  y: number
}

// const ConveyorBeltItem = WithCollisionDetection()

@WithCollisionDetection
export class ConveyorBeltItem extends FYGE.Container {
  public speed: number = 0
  public constructType: any
  propsData: ConveyorItemConstrutType
  uniqueKey: UniqueKeyType

  constructor(props: ConveyorItemConstrutType) {
    super()

    this.uniqueKey = props.uniqueKey
    this.constructType = props.Container
    this.propsData = props
    this.onRenderObstack(props)

    const _collisionConfig = props.collisionConfig
    if (_collisionConfig) { //addPythicCont
      if (_collisionConfig.type === 'rect') {
        // @ts-expect-error
        this.addPythicCont(new RectPhysicsCont(_collisionConfig.x, _collisionConfig.y, _collisionConfig.w, _collisionConfig.h))
      }
    }
  }

  reset() {
    this.speed = 0
    this.scale.set(0, 0)
    this.visible = true
  }

  get fx() {
    return this._x
  }

  set fx(v) {
    this._x = v
    this.x = this._x - (this.propsData.anchorX || 0)
  }

  get fy() {
    return this._y
  }

  set fy(v) {
    this._y = v
    this.y = this._y - (this.propsData.anchorY || 0)
  }

  _x: number = 0 // should be x axis value
  _y: number = 0
  setPosition(x: number, y: number) {
    this.fx = x
    this.fy = y
  }

  /**
   * 渲染障碍物
   * @param config 
   */
  onRenderObstack(config: ConveyorItemConstrutType) {
    const type = config.Container
    // console.log('onRenderObstacle', type)
    let ObstackIns: FYGE.Container
    if (typeof type === 'string') {
      const spotType = type.split('.')[1]
      const isImage = ['png', 'jpg', 'jpeg'].includes(spotType)
      if (isImage) {
        ObstackIns = new FYGE.Sprite(RES.getRes(type));
        // 默认居中吧
        (ObstackIns as FYGE.Sprite).anchorTexture.set(0.5, 0.5)
      } else if (spotType === 'svga') {
        ObstackIns = new FYGE.MovieClip(RES.getRes(type));
        this.anchor.set(this.propsData.anchorX || 0, this.propsData.anchorY || 0)
      }
    } else {
      ObstackIns = new type()
      this.anchor.set(this.propsData.anchorX || 0, this.propsData.anchorY || 0)
    }

    if (ObstackIns) {
      ObstackIns.visible = true
      this.addChild(ObstackIns)
    }
    
  }
}

type ProbabilityItemType = {
  value: number
  index: number
}

/**
 * 传送带
 */
export class ConveyorBelt extends FYGE.Container {
  private startPos: PosPoint
  private endPos: PosPoint
  private tanValue: number // tanValue: number
  private totalDisY: number // y 方向上的移动距离
  private clockIns: Clock

  private _ty: number // 当前y 方向上的移动距离
  private totalY: number // 当前传送带 Y 移动距离
  private tyl: number // _ty 大于 tyl 就会出新的障碍物
  private props: ConveyorBeltType

  private ProbabilityList: ProbabilityItemType[] // 障碍物的 概率数组 是经过 前置相加处理过的
  
  isMoving: boolean = false
  private _tSpeed: number
  private acc: number = 0 // 加速度
  private scaleEndProportion: number

  private ParkourGameEleList: ParkourGameEleType[]

  conveyorItemList: ConveyorBeltItem[] = []

  private onJudgeNextObstacle: (item: ParkourGameEleType) => boolean

  get tSpeed() {
    return this._tSpeed
  }

  set tSpeed(value) {
    if (value < 0) return
    this._tSpeed = value
    var t = this.totalDisY / this.tSpeed;
    this.acc = 2 * this.totalDisY / Math.pow(t, 2);
  }

  constructor(props: ConveyorBeltType, onJudgeNextObstacle?: (item: ParkourGameEleType) => boolean) {
    super()

    this.props = props

    this.onJudgeNextObstacle = onJudgeNextObstacle

    this.startPos = props.startPos
    this.endPos = props.endPos
    this.tanValue = Math.abs(props.startPos.x - props.endPos.x) ? (props.startPos.y - props.endPos.y) / (props.startPos.x - props.endPos.x) : 0
    this.totalDisY = Math.abs(props.startPos.y - props.endPos.y)
    this.clockIns = new Clock()

    if (ParkourGameConfig.CommonConfig.CollisionDebug) {
      const a = UI.Shape(this)
        .closePath()
        .beginStroke(0x00ff00)
        .lineTo(props.startPos.x, props.startPos.y)
        .lineTo(props.endPos.x, props.endPos.y)
        .endStroke()
        .closePath()
    }

    this.initEvents()
  }

  /**
   * 初始化游戏元素配置
   */
  onInitParkourGameEleConfig(e: {data: ParkourGameEleType[]}) {
    const arr = e.data
    this.ParkourGameEleList = arr

    const _ParkourGameEleList = this.ParkourGameEleList
    const sumProbability = _ParkourGameEleList.reduce((prev, cur) => prev + cur.ProbabilityPort, 0)
    const _ProbabilityList: ProbabilityItemType[]  = []
    _ParkourGameEleList.reduce((pre, curr, index) => {
      _ProbabilityList.push({
        value: pre + curr.ProbabilityPort / sumProbability,
        index
      })
      return pre + curr.ProbabilityPort / sumProbability
    }, 0)
    this.ProbabilityList = _ProbabilityList
    console.log('ProbabilityList ==== > ', _ProbabilityList)
  }

  /**
   * 初始化速度
   * @param num 
   */
  onInitTspeed(e?: {data: number}) {
    console.log(e)
    this.tSpeed = (e && e.data) || ParkourGameConfig.CommonConfig.speed || 300
  }

  /**
   * 初始化
   */
  onReset() {
    this.totalY = 0
    this._ty = 0
    this.scaleEndProportion = this.props.scaleEndProportion || 1

    this.tyl = this.props.startTyl || 0
    this.isMoving = false
    
    for (let i = this.conveyorItemList.length - 1; i >= 0; i--) {
      this.onTakeIntoPool(this.conveyorItemList[i] as ConveyorBeltItem)
    }

    this.onInitParkourGameEleConfig({
      data: this.props.ParkourGameEleList
    })

    this.onInitTspeed()
  }

  onRestartRoll() {
    this.onReset()

    this.isMoving = true
  }

  /**
   * 游戏重新开始
   */
  onGameResume() {
    this.isMoving = true
  }

  /**
   * 暂停
   */
  onRollPause() {
    this.isMoving = false
  }

  /**
   * 初始化事件
   */
  initEvents() {
    this.addEventListener(FYGE.Event.ENTER_FRAME, this.onEnterFrame, this)
    GDispatcher.addEventListener(ParkourGameEvents.GAME_RESET_PARKOURELE, this.onInitParkourGameEleConfig, this)

    GDispatcher.addEventListener(ParkourGameEvents.GAME_SPEEDUP, this.onInitTspeed, this)

    GDispatcher.addEventListener(ParkourGameEvents.GAME_PAUSE, this.onRollPause, this)

    GDispatcher.addEventListener(ParkourGameEvents.GAME_RESUME, this.onGameResume, this)

    GDispatcher.addEventListener(ParkourGameEvents.GAME_RESTART, this.onRestartRoll, this)

    GDispatcher.addEventListener(ParkourGameEvents.GAME_RESET, this.onReset, this)
  }

  /**
   * 移除事件
   */
  removeEvents() {
    this.removeEventListener(FYGE.Event.ENTER_FRAME, this.onEnterFrame, this)
    GDispatcher.removeEventListener(ParkourGameEvents.GAME_RESET_PARKOURELE, this.onInitParkourGameEleConfig, this)

    GDispatcher.removeEventListener(ParkourGameEvents.GAME_SPEEDUP, this.onInitTspeed, this)

    GDispatcher.removeEventListener(ParkourGameEvents.GAME_PAUSE, this.onRollPause, this)

    GDispatcher.removeEventListener(ParkourGameEvents.GAME_RESUME, this.onGameResume, this)

    GDispatcher.removeEventListener(ParkourGameEvents.GAME_RESTART, this.onRestartRoll, this)

    GDispatcher.removeEventListener(ParkourGameEvents.GAME_RESET, this.onReset, this)
  }

  /**
   * 
   * @param ny y方向上的移动距离
   */
  onChangePosByY(ny: number) {
    this._ty += ny
    this.totalY += ny
    // 抛出事件
    GDispatcher.dispatchEvent(ParkourGameEvents.CONVEYOR_MOVE_Y, this.totalY)
    
    if (this._ty >= this.tyl) {
      this._ty = 0
      // 可以在这里 修改 tyl
      this.onInitObstack()
    }
  }

  /**
   * 障碍物视觉移动距离
   * @param t 
   */
  onVisibleRealMove(t: number) {
    this.conveyorItemList.forEach((item: ConveyorBeltItem) => {
      var _y = item.speed * t;
      var _x = this.tanValue === 0 ? 0 : _y / this.tanValue;
      item.setPosition(item._x + _x, item._y + _y);
      var yp = _y / (this.totalDisY * this.scaleEndProportion);
      item.scaleX < 1 && (item.scaleX += yp);
      item.scaleX < 1 && (item.scaleY += yp);
      if (item.y >= this.endPos.y) {
        this.onTakeIntoPool(item);
      }
      item.speed += this.acc * t;
    })
  }

  onGetFinalObstacle(portList: ProbabilityItemType[]) {
    if (portList.length === 0) return
    const maxProbability = portList[portList.length - 1].value
    const random = Math.random() * maxProbability
    const randomItem = portList.find((item) => {
      return random <= item.value
    })
    const index = randomItem.index
    const ParkourGameEle = this.ParkourGameEleList[index]

    if (this.onJudgeNextObstacle) {
      const b = this.onJudgeNextObstacle(ParkourGameEle)
      if (!b) {
        const skipPortList = portList.filter((item) => {
          return item.index !== index
        })
        return this.onGetFinalObstacle(skipPortList)
      }
    }
    return ParkourGameEle
  }

  /**
   * 初始化障碍物
   */
  onInitObstack() {
    const ParkourGameEle = this.onGetFinalObstacle(this.ProbabilityList)

    if (!ParkourGameEle) return

    this.onTakeoutFromPool(ParkourGameEle)

    // 生成新的障碍物事件
    GDispatcher.dispatchEvent(ParkourGameEvents.CONVEYOR_INIT_OBSTACK, ParkourGameEle)
  }

  /**
   * 优先从对象池中取
   * @param GameEleConfig 
   * @returns 
   */
  onTakeoutFromPool(GameEleConfig: ParkourGameEleType) {
    const t = GPool.takeOut<ConveyorBeltItem>(GameEleConfig.Container) || new ConveyorBeltItem(GameEleConfig)
    t.setPosition(this.startPos.x, this.startPos.y)

    this.addChild(t)

    this.conveyorItemList.push(t)

    t.reset()
    return t
  }

  /**
   * 清除清除障碍物 并加到对象池
   * @param t 
   */
  onTakeIntoPool(t: ConveyorBeltItem, natural: boolean = true) {
    t.visible = false
    GPool.takeIn(t.constructType, t.parent.removeChild(t))

    this.conveyorItemList.splice(this.conveyorItemList.indexOf(t), 1)

    if (!natural) {
      GDispatcher.dispatchEvent(ParkourGameEvents.GAME_COLLESION, t)
    }
  }

  onEnterFrame() {
    if (!this.isMoving) return

    const diff =  this.clockIns.getDelta()
    if (diff > 0.1) return

    const currSpeed = this.tSpeed
    this.onChangePosByY(diff * currSpeed)

    this.onVisibleRealMove(diff)
  }

  destroy(): void {
    this.removeEvents()

    super.destroy()
  }
}