import decamelize from 'decamelize';
import crypto from 'crypto';
import babel from '@babel/core';

/**
 * Created by rockyl on 2019-11-30.
 */

const UglifyJS = require('uglify-js');

async function compile(source, debug = false) {
	const {code, map} = await babel.transformAsync(source, {
		presets: [
			['@babel/env', {}]
		],
		babelrc: false,
		sourceMaps: debug,
	});

	const result = {
		code,
	};
	if (map) {
		result.sourcemap = map.mappings;
	}
	return result;
}

function uglify(source){
	const uglifyResult = UglifyJS.minify(source, {
	});
	if (!uglifyResult.error) {
		return uglifyResult.code;
	}
}

/**
 * Created by rockyl on 2019-11-29.
 */

function md5(source) {
	const hash = crypto.createHash('md5');
	return hash.update(source).digest('hex');
}

class ProcessManager {
	constructor() {
		this.pool = {};
		this.scripts = '';
	}

	deal(process) {
		let hash = this.put(process);
		if (hash) {
			process.script = 'link://' + hash;
		}
		if (process.metas) {
			for (let subProcess of process.metas) {
				let hash = this.put(subProcess);
				if (hash) {
					subProcess.script = 'link://' + hash;
				}
			}
		}
	}

	put(process) {
		const {id, name, script} = process;
		if (script) {
			let hash = md5(script);
			if (!this.pool.hasOwnProperty(hash)) {
				this.pool[hash] = {
					script,
					ids: [],
				};
			}
			this.pool[hash].ids.push(name || id);
			return hash;
		}
	}

	generateCurrent() {
		let scripts = '';
		for (let hash in this.pool) {
			let {script, ids} = this.pool[hash];
			let idsComment = ids.map(id=>{return `/*== ${id} ==*/`}).join('\n');
			scripts += `
	exports['${hash}'] = function(args, props, target, global, vm){
		return new Promise(function(resolve, reject){

${idsComment}
${script}

		function next(type, payload){resolve({type: type, payload: payload})}
		});
	};
`;
		}
		this.scripts += scripts;
		this.pool = {};
	}

	async compile(){
		try {
			const {code ,sourcemap} = await compile(this.scripts, true);
			this.scripts = code;
		}catch (e) {
			console.log('编译失败', e);
		}
	}

	generate() {
		this.generateCurrent();
		let scripts = this.scripts;

		return `
(function(){
var exports = {};
${scripts}

	engine.setScriptMap(exports);
})();
`
	}
}

/**
 * Created by rockyl on 2019-11-29.
 */

class ScriptManager {
	constructor() {
		this.pool = [];
	}

	deal(script){
		this.pool.push(script);
	}

	generate(){
		let scripts = '';
		for(let {id, code} of this.pool){
			scripts += `
/*=====START ${id} START=====*/
(function(exports){
${code}
})(exports);
	engine.registerScriptDef(exports.default.id, exports.default);
/*=====END ${id} END=====*/`;
		}

		return `
(function(){
var exports = {};
${scripts}
})();
`;
	}
}

/**
 * Created by rockyl on 2019-11-29.
 */

class CustomManager {
	constructor() {
		this.pool = [];
	}

	deal(custom) {
		this.pool.push(custom);
	}

	generate(){
		let scripts = '';
		for(let {id, code} of this.pool){
			scripts += `
/*=====START ${id} START=====*/
module = {
	exports: {}
};
(function(module){
${code}
})(module);
exports['${id}'] = module.exports;
/*=====END ${id} END=====*/`;
		}

		return `
(function(){
var exports = {};
var module;

${scripts}

	for(var key in exports){
		engine.registerCustomModule(key, exports[key]);
	}
})();
`
	}
}

/**
 * Created by rockyl on 2019-11-13.
 *
 * 项目打包
 */

const replaceFields = ['pageTitle', 'containerId'];
const TAG = 'zeroing-pack';

async function pack(data, options) {
	let version = Date.now() + Math.floor(Math.random() * 1000);
	pageTemplate(data, data.options, version);
	const newData = await packData(data, options);

	return {
		version,
		data: newData,
	}
}

function fillTpl(data, params) {
	const {options} = data;
	fillTemplate(options.newTpl, options, params);

	return options.newTpl;
}

async function packData(data, {debug, getProcesses, getScripts, getCustoms}) {
	const processManager = new ProcessManager();
	const scriptManager = new ScriptManager();
	const customManager = new CustomManager();

	let newData = {};
	newData.options = data.options;
	newData.views = data.views;
	newData.assets = data.assets;
	newData.dataMapping = data.dataMapping;
	newData.processes = data.processes;

	delete newData.options.tpl;
	deleteUnusedData(newData.processes);

	console.log(TAG, 'start');

	/*=====START process =====*/
	console.log(TAG, 'start process');
	let processIDs = [];
	findDepPidsBat(processIDs, newData.processes);
	let builtinProcesses = newData.builtinProcesses = [];

	let bProcessIDs = processIDs;
	while (true) {
		let newPids = await addBuiltinProcesses(builtinProcesses, bProcessIDs, getProcesses);
		bProcessIDs = [];
		for (let id of newPids) {
			if (!processIDs.includes(id)) {
				bProcessIDs.push(id);
				processIDs.push(id);
			}
		}
		if (bProcessIDs.length === 0) {
			break;
		}
	}

	for (let process of newData.processes) {
		processManager.deal(process);
	}

	console.log(TAG, 'processManager.generateCurrent()');

	processManager.generateCurrent();
	await processManager.compile(); //自定义过程先编译

	for (let process of builtinProcesses) {
		processManager.deal(process);
	}

	let processScriptContent = processManager.generate();
	//console.log(processScriptContent);
	if (!debug) {
		processScriptContent = uglify(processScriptContent);
	}
	/*=====END process =====*/

	/*=====START script =====*/
	console.log(TAG, 'start script');
	let scriptIDs = [];
	for (let view of newData.views) {
		traverseNode(view, (node) => {
			if (node.scripts && node.scripts.length > 0) {
				for (let {script} of node.scripts) {
					if (!scriptIDs.includes(script)) {
						scriptIDs.push(script);
					}
				}
			}
		});
	}
	//console.log('scriptIDs:', scriptIDs);
	//let scriptsContainer = newData.scripts = {};
	//let scriptsCode = '';
	if (scriptIDs.length > 0) {
		const scripts = await getScripts(scriptIDs);
		for (let scriptData of scripts) {
			let script = JSON.parse(scriptData);
			//scriptsContainer[id] = code;
			scriptManager.deal(script);
		}
		//console.log('scripts:', scriptsContainer);
	}

	let scriptsContent = scriptManager.generate();
	//console.log(scriptsContent);
	if (!debug) {
		scriptsContent = uglify(scriptsContent);
	}
	/*=====END script =====*/

	/*=====START custom =====*/
	console.log(TAG, 'start custom');
	//newData.customs = [];
	if (data.customs && data.customs.length > 0) {
		/*newData.customs = */
		(await getCustoms(data.customs)).map(item => {
			customManager.deal(JSON.parse(item));
			//return JSON.parse(item);
		});
	}
	let customScriptContent = customManager.generate();
	//console.log(customScriptContent);
	if (!debug) {
		customScriptContent = uglify(customScriptContent);
	}
	/*=====END custom =====*/

	return {
		data: JSON.stringify(newData),
		processScriptContent,
		scriptsContent,
		customScriptContent,
	};
}

const unusedFields = ['design'];
function deleteUnusedData(processes){
	for(let process of processes){
		if(process.sub){
			for(let uuid in process.sub){
				let subProcess = process.sub[uuid];
				for(let field of unusedFields){
					if(subProcess.hasOwnProperty(field)){
						delete subProcess[field];
					}
				}
			}
		}
		if(process.metas && process.metas.length > 0){
			deleteUnusedData(process.metas);
		}
	}
}

function findDepPids(list, process) {
	if (process.sub) {
		for (let key in process.sub) {
			let p = process.sub[key];
			if (!list.includes(p.meta)) {
				list.push(p.meta);
			}
		}
	}
}

function findDepPidsBat(list, processes) {
	for (let process of processes) {
		findDepPids(list, process);
	}
}

async function addBuiltinProcesses(list, ids, getProcesses) {
	let newPids = [];
	if (ids.length > 0) {
		let processes = await getProcesses(ids);
		for (let processData of processes) {
			let process = JSON.parse(processData);
			list.push(process);
			findDepPids(newPids, process);
		}
	}
	return newPids;
}

function pageTemplate(tpl, options, version) {
	const params = {
		version,
	};
	for (let field of replaceFields) {
		params[field] = options[field];
	}
	fillTemplate(options.tpl, options, params);
}

function fillTemplate(tpl, options, params) {
	for (let field in params) {
		const pattern = decamelize(field).toUpperCase();
		tpl = tpl.replace(new RegExp(`\\$${pattern}\\$`, 'g'), params[field]);
	}

	options.newTpl = tpl;
}

function traverseNode(root, callback) {
	callback(root);
	if (root.children && root.children.length > 0) {
		for (let childNode of root.children) {
			traverseNode(childNode, callback);
		}
	}
}

export { fillTpl, pack };
//# sourceMappingURL=index.es.js.map
