/**
 * Created by rockyl on 2019-11-08.
 */
import Container from "../../2d/display/Container";

export const ESCAPE_REG_EXP = /\$\{[\u4e00-\u9fa5_a-zA-Z0-9\|]+\}/g;
export const linkedFlag = '$_linked_$';
export const nodeScheme = 'node://';
export const linkScheme = 'link://';

/**
 * 数组查找
 * @param arr
 * @param predicate
 */
export function arrayFind(arr, predicate) {
	if (!arr) {
		return;
	}
	for (let i = 0, li = arr.length; i < li; i++) {
		const item = arr[i];

		if (predicate(item, i, arr)) {
			return item;
		}
	}
}

/**
 * 对象深度克隆
 * @param obj
 */
export function objClone(obj) {
	return JSON.parse(JSON.stringify(obj));
}

const percentKeys = ['width', 'height'];

export function propertyParse(key, node, properties) {
	let value = properties[key];
	let targetKey = key;
	if (percentKeys.indexOf(key) >= 0) {
		if (typeof value === 'string') {
			if (value[value.length - 1] === '%') {
				targetKey = 'percent' + key[0].toUpperCase() + key.substr(1);
			}
			value = parseInt(value);
		}
	}
	node[targetKey] = value;
}

/**
 * 根据路径获取数据
 * @param scope
 * @param path
 * @param throwException
 */
export function getDataByPath(scope, path, throwException?) {
	let func = new Function('scope', `return scope` + (path ? ('.' + path) : ''));
	try {
		return func(scope);
	} catch (e) {
		//console.warn(e);
		if (throwException) {
			throw e;
		}
	}
}

/**
 * 安全的eval方法
 * @param code
 * @param throwException
 */
export function safeEval(code, throwException = false){
	let func = new Function(code);
	try {
		return func();
	}catch (e) {
		if (throwException) {
			throw e;
		}
	}
}

/**
 * 属性注入方法
 * @param target 目标对象
 * @param data 被注入对象
 * @param callback 自定义注入方法
 * @param ignoreMethod 是否忽略方法
 * @param ignoreNull 是否忽略Null字段
 *
 * @return 是否有字段注入
 */
export function injectProp(target: any, data?: any, callback?: Function, ignoreMethod: boolean = true, ignoreNull: boolean = true): boolean {
	if (!target || !data) {
		return false;
	}

	let result = false;
	for (let key in data) {
		let value: any = data[key];
		if ((!ignoreMethod || typeof value != 'function') && (!ignoreNull || value != null)) {
			if (callback) {
				callback(target, key, value);
			} else {
				try {
					target[key] = value;
				} catch (e) {

				}
			}

			result = true;
		}
	}
	return result;
}

/**
 * 属性拷贝
 * @param target
 * @param data
 * @param config
 */
export function copyProp(target, data, config?) {
	if (data) {
		for (let key in config) {
			let valueConfig = config[key];
			if (Array.isArray(valueConfig)) {
				target[key] = {};
				for (let field of valueConfig) {
					target[key][field] = data[key][field];
				}
			} else if (typeof valueConfig === 'string') {
				target[valueConfig] = data[valueConfig];
			} else if (typeof valueConfig === 'object') {
				target[key] = {};
				copyProp(target[key], data[key], valueConfig)
			}
		}
	}
}

/**
 * 对象转query字符串
 * @param obj
 */
export function obj2query(obj: any): string {
	if (!obj) {
		return '';
	}
	let arr: string[] = [];
	for (let key in obj) {
		arr.push(key + (key ? '=' : '') + obj[key]);
	}
	return arr.join('&');
}

function requireForCJS(id) {
	return window[id];
}

/**
 * 导入cjs包装的代码
 * @param code
 * @param node
 */
export function importCJSCode(code, node?) {
	if (node) {
		let create = new Function('module', 'require', code);
		let module = {
			exports: {},
		};
		create(module, requireForCJS);
		return module.exports;
	} else {
		let create = new Function('exports', code);
		let exports: any = {};
		create(exports);
		return exports.default;
	}
}

/**
 * 导入umd包装的代码
 * @param code
 */
export function importUMDCode(code) {
	let create = new Function('exports', code);
	let exports: any = {};
	create.call(exports);
	return exports;
}

/**
 * 字符串两端删除字符
 * @param str
 * @param char
 */
export function trimChar(str: string, char: string) {
	if (!str || !char) {
		return;
	}
	while (true) {
		let len = str.length, from = 0, end = len;
		if (str[0] === char) {
			from = 1;
		}
		if (str[len - 1] === char) {
			end = len - 1;
		}
		str = str.substring(from, end);
		if (str.length == len) {
			break;
		}
	}

	return str;
}

/**
 * 路径拼接
 * @param segments
 */
export function joinPath(...segments: string[]) {
	let result = [];
	for (let segment of segments) {
		result.push(trimChar(segment, '/'));
	}
	return result.join('/');
}

/**
 * 属性查找
 * @param name
 * @param contexts
 */
export function findVariable(name: string, ...contexts) {
	let result;
	for (let context of contexts) {
		if (context) {
			result = context[name];
			if (result !== undefined) {
				break;
			}
		}
	}
	return result;
}

let el;

/**
 * html文本转纯文本
 * @param htmlText
 */
export function htmlToPureText(htmlText) {
	if (!el) {
		el = document.createElement('div');
	}
	el.innerHTML = htmlText;
	document.body.appendChild(el);
	let pureText = el.innerText;
	document.body.removeChild(el);

	return pureText;
}

const zhReg = /[\u4e00-\u9fa5]/;

/**
 * 字符串长度，中文占2个字母宽度
 * @param str
 */
export function strLen(str) {
	let len = 0;
	for (let char of str) {
		len += char.match(zhReg) ? 2 : 1;
	}
	return len;
}

/**
 * 字符串简短化
 * @param str
 * @param limit
 * @param replace
 */
export function strShort(str, limit, replace = '…') {
	let result = '';
	if (strLen(str) > limit) {
		let len = 0;
		for (let i = 0, li = str.length; i < li; i++) {
			const char = str[i];
			len += char.match(zhReg) ? 2 : 1;
			if (len > limit) {
				result += replace;
				break;
			} else {
				result += char;
			}
		}
	} else {
		result = str;
	}
	return result;
}

/**
 * 脚本实例化
 * @param node
 * @param ScriptConfig
 */
export function instantiateScript(node, ScriptConfig) {
	const {script: scriptName, props, disabled} = ScriptConfig;
	const script = node.scripts.add(scriptName, props, disabled);
}

export const cmdPrefix = 'z-';
export const cmdOldPrefix = '//z-';

/**
 * 属性注入
 * @param target
 * @param source
 */
export function injectProperties(target, source) {
	for (let key in source) {
		if(key.indexOf('//') !== 0 && !source.hasOwnProperty(cmdPrefix + key)){
			propertyParse(key, target, source);
		}
	}

	return target;
}

/**
 * 遍历视图节点
 * @param node
 * @param callback
 * @param includeSelf
 */
export function traverseViewNode(node: Container, callback: Function, includeSelf = false) {
	if(includeSelf){
		callback(node);
	}
	for (let child of node.children) {
		callback(child, node);

		if (child.children && child.children.length > 0) {
			traverseViewNode(child, callback);
		}
	}
}
