/**
 * Created by rockyl on 2019-12-13.
 */
import {Label} from "./Label";
import {MouseEvent} from "../../../2d/events/MouseEvent";
import {Event} from "../../../2d/events/Event";
import {FloatDisplay} from "../../../2d/display/FloatDisplay";
import {TextField} from "../../../2d/text/index";
import {Point} from "../../../2d/math/index";
import {dirtyFieldTrigger} from "../../decorators";
import {VERTICAL_ALIGN} from "../../../2d/const";
import {Container} from "../../../2d/display/index";
import {IUIComponent} from "./IUIComponent";

let timer;

function delayScrollTop() {
	cancelDelayScrollTop();
	timer = setTimeout(function () {
		window.scrollTo(0, 0);
	}, 100);
}

function cancelDelayScrollTop() {
	if (timer) {
		clearTimeout(timer);
		timer = null;
	}
}

export class TextInput extends Label implements IUIComponent {
	isUI = true;

	private _floatDisplay: FloatDisplay;
	private _placeholderLabel: TextField;
	private _input: any;
	@dirtyFieldTrigger
	placeholder: string;
	@dirtyFieldTrigger
	placeholderColor: any = '#666666';
	@dirtyFieldTrigger
	maxLength: number;
	@dirtyFieldTrigger
	type: string = 'text';
	@dirtyFieldTrigger
	charRegStr: string;

	private _oldFillColor;
	private _oldStrokeColor;
	private _charReg: RegExp;

	constructor() {
		super();

		this.setup();
		this.addEventListener(MouseEvent.CLICK, this.onClickStatic, this);
		this.addEventListener(Event.RESIZE, this.onResize, this);

		this.text = '';
	}

	onModify(value, key) {
		switch (key) {
			case 'placeholder':
				if (this._placeholderLabel) {
					this._placeholderLabel.text = value;
				}
				break;
			case 'placeholderColor':
				if (this._placeholderLabel) {
					this._placeholderLabel.fillColor = value;
				}
				break;
			case 'maxLength':
				this._text = this.updateMaxLength(this._text);
				break;
			case 'charRegStr':
				this.setCharReg();
				break;
		}
	}

	private setup() {
		let fd = this._floatDisplay = new FloatDisplay();
		let input = this._input = document.createElement('input');
		fd.htmlElement = input;
		fd.alpha = 0;
		input.style.pointerEvents = 'none';

		let style = input.style;
		style.outline = 'none';
		style.padding = '0';
		style.border = '0';
		style.backgroundColor = 'transparent';

		input.type = 'text';
		input.addEventListener('focus', this.onFocus);
		input.addEventListener('blur', this.onBlur);
		input.addEventListener('input', this.onInput);

		let pl = this._placeholderLabel = new TextField();
		pl.fillColor = this.placeholderColor;
		this.verticalAlign = pl.verticalAlign = VERTICAL_ALIGN.MIDDLE;

		this.addChild(fd);
	}

	protected _setText(value) {
		super._setText(value);

		this._text = this.updateMaxLength(this._text);
		this._text = this.updateWithCharReg(this._text);

		setTimeout(() => {
			this.showPlaceholderLabel(value)
		}, 1);
	}

	private updateMaxLength(text) {
		let value = text;
		let maxLength = this.maxLength;
		if (maxLength > 0 && value && value.length > maxLength) {
			value = value.substr(0, maxLength);
		}
		return value;
	}

	private updateWithCharReg(text) {
		let value = text;
		if (this._charReg) {
			value = '';
			for (let char of text) {
				if (char.match(this._charReg)) {
					value += char;
				}
			}
		}
		return value;
	}

	private setCharReg() {
		if (this.charRegStr) {
			this._charReg = new RegExp(this.charRegStr);
			this._text = this.updateWithCharReg(this._text);
		} else {
			this._charReg = null;
		}
	}

	private showPlaceholderLabel(value) {
		let pl = this._placeholderLabel;
		if (value) {
			let pl = this._placeholderLabel;
			if (pl.parent) {
				pl.parent.removeChild(pl);
			}
		} else {
			if (!pl.parent) {
				pl.text = this.placeholder;
				pl.size = this.size;
				pl.font = this.font;
				this.addChildAt(pl, 0);
			}
			pl.textAlign = this.textAlign;
		}
	}

	setFocus() {
		let fd = this._floatDisplay;
		let input = this._input;
		let pl = this._placeholderLabel;

		fd.width = this.width;
		fd.height = this.height;

		input.value = this.text;
		let style = input.style;
		style.fontSize = this.size + 'px';
		style.color = this.fillColor;
		style.width = this.width + 'px';
		style.height = this.height + 'px';
		style.fontFamily = this.font;
		style.pointerEvents = 'auto';
		style.textAlign = this.textAlign;

		this._oldFillColor = this.fillColor;
		this._oldStrokeColor = this.strokeColor;
		this.fillColor = 'transparent';
		this.strokeColor = 'transparent';

		if (pl.parent) {
			pl.parent.removeChild(pl);
		}

		//this.addChild(this._floatDisplay);

		this._floatDisplay.alpha = 1;

		const maxLength = this.maxLength;
		if (maxLength > 0) {
			input.maxLength = maxLength;
		} else {
			input.removeAttribute('maxLength')
		}

		if (this.type) {
			input.type = this.type;
		}

		input.focus();

		this.dispatchEvent(Event.FOCUS);
	}

	setBlur() {
		let input = this._input;

		this._floatDisplay.alpha = 0;
		input.style.pointerEvents = 'none';

		input.blur();

		this.text = input.value;

		this.fillColor = this._oldFillColor;
		this.strokeColor = this._oldStrokeColor;

		this.dispatchEvent(Event.BLUR);
	}

	private onFocus = (e) => {
		cancelDelayScrollTop();
	};

	private onBlur = (e) => {
		this.setBlur();

		delayScrollTop();
	};

	private onInput = (e) => {
		if (this._charReg) {
			//console.log('before:', this._input.value);
			let text = this.updateWithCharReg(this._input.value);
			//console.log('after:', text);
			this._input.value = text;
		}
		this.dispatchEvent(Event.CHANGING, this._input.value);
	};

	private onClickStage(e) {
		//e.stopPropagation();
		if (e.currentTarget !== this) {
			this.setBlur();
		}

	}

	private onClickStatic() {
		this.setFocus();
		this.stage.once(MouseEvent.MOUSE_DOWN, this.onClickStage, this);
	}

	private onResize() {
		let pl = this._placeholderLabel;
		pl.width = this.width;
		pl.height = this.height;
	}

	public static _bp: Point = new Point();

	/**
	 * 检测点是否在任何child上
	 * 重写父类方法
	 */
	hitTestPoint(globalPoint: Point, isMouseEvent: boolean = false) {
		const s = this;
		//如果不可见
		if (!this.visible) return null;
		//如果禁止子级的鼠标事件
		if (isMouseEvent && !this.mouseChildren) return null;
		if (!isMouseEvent) {
			//如果不是系统调用则不考虑这个点是从全局来的，只认为这个点就是当前要碰撞测试同级别下的坐标点
			if (s._localBoundsSelf.isPointIn(globalPoint)) {
				return s;
			}
		} else {
			let {x, y} = s.globalToLocal(globalPoint, TextInput._bp);
			if (x > 0 && x < s.width && y > 0 && y < s.height) {
				return s;
			}
		}

		return null;
	}
}
