/**
 * Created by rockyl on 2020-03-19.
 */
import {DisplayObject} from "../../../2d/display/DisplayObject";
import {ZriCompiler} from "./ZriCompiler";
import {IWatcher, IZri, ZriCommandData} from "./Interfaces";
import {evalExp, runExp} from "./Utils";
import {Container} from "../../../2d/display/index";

export interface Command {
	/**
	 * 执行命令
	 * @param context 命令上下文
	 */
	(context?: CommandContext): DisplayObject;
}

export interface CommandContext {
	scope: any;
	target: Container;
	entity: IZri;
	cmdData: ZriCommandData;

	[name: string]: any;
}

export const commands: { [name: string]: Command } = {
	prop: (context: CommandContext) => {
		let cmdData: ZriCommandData = context.cmdData;
		let target: DisplayObject = context.target;
		context.entity.createWatcher(target, cmdData.exp, context.scope, (value: any) => {
			if (cmdData.subCmd != "") {
				target[cmdData.subCmd] = value;
			} else {
				for (let name in value) {
					target[name] = value[name];
				}
			}
		});
		// 返回节点
		return target;
	},
	on: (context: CommandContext) => {
		let cmdData: ZriCommandData = context.cmdData;
		if (cmdData.subCmd != "") {
			let handler: Function = context.scope[cmdData.exp] || window[context.cmdData.exp];
			if (typeof handler == "function") {
				// 是函数名形式
				context.target.addEventListener(cmdData.subCmd, function () {
					handler.apply(this, arguments);
				}, context.scope);
			} else {
				// 是方法执行或者表达式方式
				context.target.addEventListener(cmdData.subCmd, (evt: Event) => {
					// 创建一个临时的子域，用于保存参数
					let scope: any = Object.create(context.scope);
					scope.$event = evt;
					scope.$target = context.target;
					runExp(cmdData.exp, scope);
				});
			}
		}
		// 返回节点
		return context.target;
	},
	if: (context: CommandContext) => {
		let cmdData: ZriCommandData = context.cmdData;
		// 记录一个是否编译过的flag
		let compiled: boolean = false;
		// 插入一个占位元素
		let refNode: Container = new Container();
		refNode.mouseChildren = refNode.mouseEnabled = false;
		let parent: Container = context.target.parent;
		let index: number = parent.getChildIndex(context.target);
		parent.removeChildAt(index);
		parent.addChildAt(refNode, index);
		// 只有在条件为true时才启动编译
		let watcher: IWatcher = context.entity.createWatcher(context.target, cmdData.exp, context.scope, (value: boolean) => {
			// 如果refNode被从显示列表移除了，则表示该if指令要作废了
			if (!refNode.parent && !context.target.parent) {
				watcher.dispose();
				return;
			}
			if (value == true) {
				// 插入节点
				if (!context.target.parent) {
					let parent = refNode.parent;
					let index: number = parent.getChildIndex(refNode);
					parent.removeChild(refNode);
					parent.addChildAt(context.target, index);
				}
				// 启动编译
				if (!compiled) {
					context.compiler.compile(context.target, context.scope);
					compiled = true;
				}
			} else {
				// 移除元素
				if (context.target.parent) {
					let parent = context.target.parent;
					let index: number = parent.getChildIndex(context.target);
					parent.removeChild(context.target);
					parent.addChildAt(refNode, index);
				}
			}
		});
		// 返回节点
		return context.target;
	},
	for: (context: CommandContext) => {
		let cmdData: ZriCommandData = context.cmdData;
		let options = evalExp(cmdData.subCmd, context.scope) || {};
		let page: number = (options.page || Number.MAX_VALUE);
		// 解析表达式
		let reg: RegExp = /^\s*(\S+)\s+in\s+([\s\S]+?)\s*$/;
		let res: RegExpExecArray = reg.exec(cmdData.exp);
		if (!res) {
			console.error("for命令表达式错误：" + cmdData.exp);
			return;
		}
		let itemName: string = res[1];
		let arrName: string = res[2];
		// 生成一个容器替换原始模板
		let index: number = context.target.parent.getChildIndex(context.target);
		let parent: Container = new Container();
		context.target.parent.addChildAt(parent, index);
		context.target.parent.removeChild(context.target);
		// 生成一个新的scope，要向其中添加属性
		let forScope: any = Object.create(context.scope);
		Object.defineProperty(forScope, "$forTarget", {
			configurable: true,
			enumerable: false,
			value: context.target,
			writable: false
		});
		// 如果有viewport命令，则将其转移至容器上
		/*let viewportCmds:ZriCommandData[] = context.cmdDict["viewport"];
		if(viewportCmds)
		{
			let viewportCmd:ZriCommandData = viewportCmds[0];
			if(viewportCmd)
			{
				parent[viewportCmd.propName] = viewportCmd.exp;
				delete context.target[viewportCmd.propName];
			}
		}*/
		// 使用原始显示对象编译一次parent
		context.compiler.compile(parent, forScope);
		// 获取窗口显示范围
		//let viewportHandler:ViewPortHandler = getViewportHandler(parent);
		// 声明闭包数据
		let isArray: boolean;
		let curList: any[];
		let curIndex: number;
		let lastNode: DisplayObject;
		// 添加订阅
		let watcher: IWatcher = context.entity.createWatcher(context.target, arrName, forScope, (value: any) => {
			// 如果refNode被从显示列表移除了，则表示该for指令要作废了
			if (!parent.parent) {
				watcher.dispose();
				return;
			}
			// 清理原始显示
			for (let i: number = parent.children.length - 1; i >= 0; i--) {
				parent.removeChildAt(i).destroy();
			}
			// 如果是数字，构建一个数字列表
			if (typeof value == "number") {
				let temp: number[] = [];
				for (let i: number = 0; i < value; i++) {
					temp.push(i);
				}
				value = temp;
			}
			// 如果不是数组，而是字典，则转换为数组，方便中断遍历
			isArray = (value instanceof Array);
			let list: any[];
			if (isArray) {
				list = value;
			} else {
				list = [];
				for (let key in value) {
					list.push({
						key: key,
						value: value[key]
					});
				}
			}
			// 初始化数据
			curList = list;
			curIndex = 0;
			lastNode = null;

			for (let li = curList.length; curIndex < li; curIndex++) {
				//渲染
				// 拷贝一个target
				let newNode: DisplayObject = context.target.clone(true, true);
				// 添加到显示里
				parent.addChild(newNode);
				// 生成子域
				let newScope: any = Object.create(forScope);

				// 这里一定要用defineProperty将目标定义在当前节点上，否则会影响forScope
				Object.defineProperty(newScope, "$index", {
					configurable: true,
					enumerable: false,
					value: curIndex,
					writable: false
				});
				// 如果是字典则额外注入一个$key
				if (!isArray) {
					Object.defineProperty(newScope, "$key", {
						configurable: true,
						enumerable: true,
						value: curList[curIndex].key,
						writable: false
					});
				}
				// 注入上一个显示节点
				Object.defineProperty(newScope, "$last", {
					configurable: true,
					enumerable: false,
					value: lastNode,
					writable: false
				});
				// 添加长度
				Object.defineProperty(newScope, "$length", {
					configurable: true,
					enumerable: false,
					value: curList.length,
					writable: false
				});
				// 注入遍历名
				Object.defineProperty(newScope, itemName, {
					configurable: true,
					enumerable: true,
					value: (isArray ? curList[curIndex] : curList[curIndex].value),
					writable: false
				});
				// 把单项数据注入到显示对象上
				Object.defineProperty(newNode, "$data", {
					configurable: true,
					enumerable: false,
					value: (isArray ? curList[curIndex] : curList[curIndex].value),
					writable: false
				});
				// 开始编译新节点
				context.compiler.compile(newNode, newScope);
				// 赋值上一个节点
				lastNode = newNode;
			}
		});
		// 返回节点
		return context.target;
	}
};