/**
 * Created by rockyl on 2019-11-08.
 *
 * 数据中心
 */
import {EventDispatcher} from "../../2d/events/index";
import {arrayFind, getDataByPath} from "../utils/index";
import {DATA_CENTER_EVENT, globalEvent} from "../decorators/events";

/**
 * 数据中心
 */
export class DataCenter extends EventDispatcher {
	store = {};
	dataMappings = [];

	constructor() {
		super();
	}

	/**
	 * 注册一个分组
	 * @param name
	 * @param origin
	 */
	registerGroup(name, origin?) {
		return this.store[name] = origin === undefined ? {} : origin;
	}

	/**
	 * 反注册分组
	 * @param name
	 */
	unregisterGroup(name) {
		delete this.store[name];
	}

	/**
	 * 获取分组
	 * @param name
	 */
	getGroup(name) {
		return this.store[name];
	}

	/**
	 * 根据路径获取数据
	 * @param path
	 * @param groupName
	 * @param throwException
	 */
	getDataByPath(path, groupName?, throwException?) {
		let scope;
		if (groupName === undefined) {
			scope = this.store;
		} else {
			let group = this.getGroup(groupName);
			scope = group === undefined ? this.store : group;
		}
		return getDataByPath(scope, path, throwException);
	}

	/**
	 * 根据绑定名获取数据
	 * @param name
	 * @param throwException
	 */
	getDataByName(name, throwException?) {
		let dataMapping = this.getDataMapping(name);
		if (dataMapping) {
			return getDataByPath(this.store, dataMapping.path, throwException);
		}
	}

	/**
	 * 填充数据
	 * @param str
	 * @param escapes
	 */
	formatString(str, escapes) {
		let result = str;
		for (let escape of escapes) {
			let args = escape.split('|');
			let name = args[0];
			let dataMapping = this.getDataMapping(name);
			try {
				let data: any = this.getDataByPath(dataMapping.path, undefined, false);
				if (args[1] !== undefined) {
					data = data[args[1]];
				}
				if (data === undefined) {
					data = '';
				}
				result = result.replace(new RegExp('\\$\\{' + escape.replace(/\|/g, '\\|') + '\\}', 'g'), data);
			} catch (e) {

			}
		}
		return result;
	}

	/**
	 * 数值递增
	 * @param groupName
	 * @param step
	 * @param path
	 * @param dispatch
	 */
	increase(groupName, step?, path?, dispatch = true) {
		if (step < 0 || step > 0) {
			let data: any = this.getDataByPath(path, groupName);
			if (data === undefined) {
				data = 0;
			} else {
				data = parseInt(data);
			}

			data += step;
			this.mutate(groupName, data, path, dispatch);
		}
	}

	/**
	 * 修改数据
	 * @param groupName
	 * @param data
	 * @param path
	 * @param dispatch
	 */
	mutate(groupName, data?, path?, dispatch = true) {
		if (!groupName) {
			return;
		}
		let group = this.getGroup(groupName);

		if (!group) {
			group = this.registerGroup(groupName);
		}
		if (data !== undefined) {
			if (path) {
				let func = new Function('scope', 'data', `scope.${path}=data`);
				try {
					func(group, data);
				} catch (e) {
					console.warn(e);
				}
			} else {
				this.registerGroup(groupName, data);
			}
		}

		if (dispatch) {
			for (let dataMapping of this.dataMappings) {
				if (dataMapping.path.indexOf(groupName) === 0) {
					this.dispatchEvent(dataMapping.name, {
						name: dataMapping.name,
						path: dataMapping.path,
						data: this.getDataByPath(dataMapping.path),
					});
				}
			}
		}

		globalEvent.dispatchEvent(DATA_CENTER_EVENT, {
			name: groupName,
			path,
			data,
		})
	}

	/**
	 * 添加数据映射
	 * @param name
	 * @param path
	 */
	dataMapping(name, path) {
		let dataMapping = {name, path};
		this.dataMappings.push(dataMapping);
	}

	/**
	 * 根据名字获取映射
	 * @param name
	 */
	getDataMapping(name) {
		return arrayFind(this.dataMappings, dataMapping => dataMapping.name === name);
	}

	/**
	 * 注册数据绑定配置
	 * @param dataMapping
	 */
	registerDataMapping(dataMapping) {
		for (let item of dataMapping) {
			this.dataMapping(item.name, item.path);
		}
	}

}

export const dataCenter = new DataCenter();
