/**
 * Created by rockyl on 2019-11-05.
 */

import {Stage} from "../../2d/display/index";
import {StackContainer} from "./StackContainer";
import {loadAssets} from "./assets-manager";
import {instantiate} from "./view-interpreter";
import {dataCenter, DataCenter} from "./data-center";
import {setProcessMetaLibs} from "../behavior-runtime/index";
import {Tween} from "../../2d/tween/index";
import {Rect} from "./nodes/index";
import {injectEnv} from "./enviroment";
import {Toast} from "./Toast";
import {arrayFind} from "../utils/index";
import {Node} from "./nodes/Node";
import {bind, createStore} from "./mvvm/index";
import {registerCustomModules} from "./custom-module";
import {dealPageRemainTime, dealPxEnv} from "../px-logics";

/**
 * 游戏舞台
 */
export class GameStage extends Node {
	private _sceneContainer: StackContainer; //场景容器
	private _popupContainer: StackContainer; //弹层容器
	private _toast: Toast;
	private _blackLayer: Rect;
	private _stage;
	private _dataCenter: DataCenter;

	private _config: any;
	private _viewCache: any = {};

	constructor(stage: Stage) {
		super();

		this.name = 'game-stage';

		stage.width = stage.viewRect.width;
		stage.height = stage.viewRect.height;
		this._stage = stage;
		this._dataCenter = dataCenter;

		this.percentWidth = 100;
		this.percentHeight = 100;
		this.mouseEnabled = false;

		let blackLayer = this._blackLayer = new Rect();

		this.addChild(this._sceneContainer = new StackContainer());
		this.addChild(blackLayer);
		this.addChild(this._popupContainer = new StackContainer());
		this.addChild(this._toast = new Toast(this));

		blackLayer.name = 'blackLayer';
		blackLayer.percentWidth = 100;
		blackLayer.percentHeight = 100;
		blackLayer.visible = false;
		blackLayer.fillColor = 0;
		blackLayer.alpha = 0.7;

		this._sceneContainer.name = 'scene-container';
		this._popupContainer.name = 'popup-container';

		this._popupContainer.addEventListener('change', this.onPopupContainerChange, this);

	}

	/**
	 * 场景容器
	 */
	get sceneContainer(): StackContainer {
		return this._sceneContainer;
	}

	/**
	 * 弹层容器
	 */
	get popupContainer(): StackContainer {
		return this._popupContainer;
	}

	/**
	 * Toast提示
	 */
	get toast(): Toast {
		return this._toast;
	}

	/**
	 * 数据中心
	 */
	get dataCenter(): DataCenter {
		return this._dataCenter;
	}

	/**
	 * 启动游戏
	 * @param config
	 * @param onAssetsProgress
	 * @param onAssetsComplete
	 */
	async launch(config, onAssetsProgress?, onAssetsComplete?, onStart?) {
		this._config = config;
		const {assets, customs} = config;
		let loaded = 0;
		let total = assets.length;
		if (customs) {
			for (let custom of customs) {
				if (custom.assets) {
					total += custom.assets.length;
				}
			}
		}

		await loadAssets(assets, p).catch(e => {
			console.log(e);
		});
		if (customs) {
			for (let custom of customs) {
				if (custom.assets) {
					await loadAssets(custom.assets, p).catch(e => {
						console.log(e);
					});
				}
			}
		}

		this.start();

		dealPxEnv();
		dealPageRemainTime();

		onStart && onStart();

		function p() {
			loaded++;

			onAssetsProgress && onAssetsProgress(loaded, total);
			if (loaded >= total) {
				onAssetsComplete && onAssetsComplete();
			}
		}
	}

	/**
	 * 开始游戏
	 */
	start() {
		const {options: {entrySceneView, env}, dataMapping, processes, builtinProcesses, scripts, customs} = this._config;

		Stage.addUpdateObj(Tween);

		injectEnv(env);

		//registerScripts(scripts);
		//registerCustomModuleFromConfig(customs);

		registerCustomModules(customs);

		if (dataMapping) {
			this.dataCenter.registerDataMapping(dataMapping);
		}
		setProcessMetaLibs(processes, builtinProcesses);

		setTimeout(() => {
			let sceneEntry = this.instantiateView(entrySceneView);
			if (sceneEntry) {
				this._sceneContainer.push(sceneEntry);
			}
		})
	}

	/**
	 * 实例化视图
	 * @param name
	 * @param cache 如果开启缓存，就会以单例形式存在
	 */
	instantiateView(name, cache = true) {
		let view;
		if (cache) {
			view = this._viewCache[name];
		}
		if (!view) {
			let viewConfig = this.getViewConfigByName(name);
			if (viewConfig) {
				view = instantiate(viewConfig);

				let store = {};
				if (viewConfig.store) {
					const {exp, computed} = viewConfig.store;
					store = createStore(exp, computed);
				}

				view['$isViewRoot'] = true;
				view['$_store'] = store;

				/*let label = view.children[0];
				label['z-for'] = 'item in list';*/

				console.time('bind');
				bind(store, view);
				console.timeEnd('bind');

				if (cache) {
					this._viewCache[name] = view;
				}
			} else {
				console.error('view config not exists:', name);
			}
		}
		return view;
	}

	/**
	 * 根据name获取视图配置
	 * @param name
	 */
	getViewConfigByName(name) {
		return arrayFind(this._config.views, view => view.name === name);
	}

	/**
	 * 设置半透明层是否可见
	 * @param visible
	 */
	setBlackLayerVisible(visible) {
		this._blackLayer.visible = visible;
	}

	onPopupContainerChange(e) {
		const {action, view, options} = e.data;

		switch (action) {
			case 'push':
			case 'replace':
			case 'popAll':
				if (options && options.center) {
					view.horizonCenter = 0;
					view.verticalCenter = 0;
				}
				break;
		}

		this.setBlackLayerVisible(this._popupContainer.children.length > 0);
	}
}
