/**
 * Created by rockyl on 2019-11-07.
 *
 * 脚本功能
 */
import {Container} from "../../2d/display";
import {Event} from "../../2d/events";
import {importCJSCode} from "../utils";

const scriptDefs = {};

/**
 * 应用脚本
 * @param ctor
 */
export function applyScript(ctor: Function) {
	ctor.prototype.applyScripts = function () {
		let scriptsProxy = this.scripts = new ScriptsProxy(this);

		this.addEventListener(Event.ENTER_FRAME, scriptsProxy.onEnterFrame, scriptsProxy);
		this.addEventListener(Event.ADDED_TO_STAGE, scriptsProxy.onAddedToStage, scriptsProxy);
		this.addEventListener(Event.REMOVED_FROM_STAGE, scriptsProxy.onRemovedFromStage, scriptsProxy);
	};
}

/**
 * 脚本接口
 */
export class ScriptBase {
	private _host: Container;
	private _disabled: boolean;

	get host() {
		return this._host
	}

	get disabled(){
		return this._disabled;
	}
	set disabled(v){
		if (this._disabled !== v) {
			this._disabled = v;

			if (this._disabled) {
				this.sleep();
			} else {
				this.awake();
			}
		}
	}

	mounted() {

	}

	destroy() {

	}

	update(t: number) {

	}

	awake() {

	}

	sleep() {

	}
}

/**
 * 注册脚本定义
 * @param id
 * @param def
 */
export function registerScriptDef(id, def) {
	scriptDefs[id] = def;

	/*Object.defineProperty(def, 'disabled', {
		get: function () {
			return !!this._disabled;
		},
		set: function (v) {
			if (this._disabled !== v) {
				this._disabled = v;

				if (this._disabled) {
					this.awake();
				} else {
					this.sleep();
				}
			}
		},
		enumerable: true,
	});*/
}

export function registerScripts(scripts) {
	for (let script of scripts) {
		let {id, code} = script;
		let def = importCJSCode(code);
		registerScriptDef(id, def);
	}
}

/**
 * 脚本容器
 */
class ScriptsProxy {
	private _host: Container;
	private _scripts: ScriptBase[] = [];

	constructor(host) {
		this._host = host;
	}

	get host() {
		return this._host;
	}

	/**
	 * 增加一个脚本
	 * @param name
	 * @param options
	 * @param disabled
	 */
	add(name, options, disabled = false): ScriptBase {
		let def = scriptDefs[name];
		if (!def) {
			console.warn(`script[${name}] def not exists`);
			return;
		}
		let script: ScriptBase = new def();
		script['_host'] = this._host;
		script['_disabled'] = disabled; //因为在注册的时候，会包装disabled属性
		for (let k in options) {
			script[k] = options[k];
		}
		this._scripts.push(script);

		script.mounted && script.mounted();
		if (this._host && this._host.stage) {
			script.awake && script.awake();
		}

		return script;
	}

	/**
	 * 移除一个脚本
	 * @param index
	 */
	remove(index): ScriptBase {
		let script = this._scripts.splice(index, 1)[0];
		if (script) {
			if (this._host && this._host.stage) {
				script.sleep && script.sleep();
			}
			script.destroy && script.destroy();
		}
		return script;
	}

	/**
	 * 获取所有脚本
	 */
	get all(): ScriptBase[] {
		return this._scripts;
	}

	/**
	 * 根据名字获取脚本
	 * @param id
	 */
	get(id): ScriptBase[] {
		return this._scripts.filter(script => {
			return script.constructor['id'] === id;
		});
	}

	/**
	 * 唤醒
	 */
	onAddedToStage() {
		for (let script of this._scripts) {
			if (!script.disabled) {
				script.awake && script.awake();
			}
		}
	}

	/**
	 * 睡眠
	 */
	onRemovedFromStage() {
		for (let script of this._scripts) {
			if (!script.disabled) {
				script.sleep && script.sleep();
			}
		}
	}

	/**
	 * 更新脚本时钟
	 */
	onEnterFrame(e) {
		let t = e.data;
		for (let script of this._scripts) {
			if (!script.disabled) {
				script.update && script.update(t);
			}
		}
	}
}
