/**
 * Created by rockyl on 2019-11-08.
 */


import {Event, EventDispatcher, MouseEvent} from "../../2d/events";
import {HashObject} from "../../2d/HashObject";
import {executeBehavior} from "../behavior-runtime";
import {injectProp} from "../utils";

const eventsMapping = {
	[Event.ADDED_TO_STAGE]: 'awake',
	[Event.REMOVED_FROM_STAGE]: 'sleep',
	[Event.UPDATE_DATA]: 'update-data',
	[MouseEvent.CLICK]: 'click',
	[MouseEvent.MOUSE_DOWN]: 'touchstart',
	[MouseEvent.MOUSE_MOVE]: 'touchmove',
	[MouseEvent.MOUSE_UP]: 'touchend',
	[Event.VISIBLE]: 'visible',
	[Event.HIDDEN]: 'hidden',
};

export const globalEvent = new EventDispatcher();

export const DATA_CENTER_EVENT: string = 'DATA_CENTER_EVENT';

const sendEvents = [
	'click',
	'touchstart',
	'touchmove',
	'touchend',
];

/**
 * 应用事件委托
 * @param ctor
 */
export function applyEvents(ctor: Function) {
	ctor.prototype.applyEvents = function () {
		let eventsProxy = this.eventsProxy = new EventsProxy(this);
		for (let k in eventsMapping) {
			this.addEventListener(k, eventsProxy.onBuiltinEvent, eventsProxy);
		}

		globalEvent.addEventListener(DATA_CENTER_EVENT, eventsProxy.onDateCenterEvent, eventsProxy);
	};
}

class EventsProxy extends HashObject {
	host;

	private _eventsConfig: any;

	constructor(host) {
		super();
		this.host = host;
	}

	get eventsConfig() {
		return this._eventsConfig;
	}

	start(eventsConfig) {
		this._eventsConfig = eventsConfig;
		for (let name in eventsConfig) {
			let config = eventsConfig[name];
			if(config.custom){
				let target = config.private ? this.host : globalEvent;

				target.addEventListener(name, this.onCustomEvent, this);
			}
		}

		this.invokeBehavior('init', this.host);
	}

	invokeBehavior(name, target, payload?) {
		if (this._eventsConfig) {
			const eventConfig = this._eventsConfig[name];
			if (eventConfig) {
				for (let behavior of eventConfig.behaviors) {
					if (behavior.disabled) {
						continue;
					}
					executeBehavior({
						main: behavior,
					}, 'main', target, payload);
				}
			}
		}
	}

	onBuiltinEvent(e: MouseEvent) {
		let eventName = eventsMapping[e.type];
		if (eventName) {
			let data = {};
			injectProp(data, e.data);
			if(sendEvents.indexOf(eventName) >= 0){
				injectProp(data, {
					localX: e.localX,
					localY: e.localY,
					stageX: e.stageX,
					stageY: e.stageY,
					identifier: e.identifier,
				});
			}
			this.invokeBehavior(eventName, e.target, data);
		}
	}

	onCustomEvent(e) {
		this.invokeBehavior(e.type, this.host, e.data);
	}

	onDateCenterEvent(e) {
		this.invokeBehavior('data-center', this.host, e.data);
	}

	destroy(): void {
	}
}
