/**
 * Created by rockyl on 2019-09-03.
 *
 * 过程
 */
import {VM} from "./VM";
import {getDataByPath, linkScheme, nodeScheme, objClone} from "../utils/index";
import {dataCenter} from "../game-warpper/data-center";
import {env} from "../game-warpper/enviroment";
import {getLogSwitch, Logs} from "../log-switch";

const logConfig = getLogSwitch(Logs.PROCESS);

function logProcess(meta, vm, process, ...params) {
	let ci = (vm.id * 65) % 360;
	const logStyle = `color: hsl(${ci},65%,75%)`;

	let showLog = false;
	if (typeof logConfig === 'boolean') {
		showLog = logConfig;
	} else if (logConfig.indexOf(meta.id) >= 0 || logConfig.indexOf(meta.name) >= 0) {
		showLog = true;
	}

	if (showLog) {
		console.log(`%c[${vm.id}:${process.id}]`, logStyle, ...params);
	}
}

export class Process {
	private readonly id;
	private _config;
	private _parent: Process;
	private _vm: VM;
	private _sequence;
	private _meta;
	private _target;
	private _originProps;
	private _scope: any;

	constructor(id) {
		this.id = id;
		this._scope = {};
	}

	get processConfig() {
		return this._config;
	}

	get parent() {
		return this._parent;
	}

	get sequence() {
		return this._sequence;
	}

	get scope() {
		return this._scope;
	}

	init(context) {
		const {vm, parent, sequence, id, target} = context;
		this._vm = vm;
		this._parent = parent;
		this._config = sequence[id];
		this._sequence = sequence;
		this._target = target;

		if (!this._config._originProps && this._config.props) {
			this._config._originProps = objClone(this._config.props)
		}
	}

	async execute(args) {
		let metaConfig = this._config.meta;
		let meta, result;
		if (metaConfig) {
			this._meta = meta = this.getProcessMeta(metaConfig);
		}
		if (!meta) {
			console.warn(`meta [${metaConfig}] is not found`);
			return;
		}
		logProcess(meta, this._vm, this, `execute [${this._config.alias || meta.name || meta.id}]`);
		if (meta.isDivider) {
			let p;
			for (let i = 0, li = meta.output.length; i < li; i++) {
				const key = meta.output[i];
				p = this._executeNextProcess(key, args);
				if (i === li - 1) {
					result = await p;
				} else {
					p.catch(e => {
						console.warn(e);
					})
				}
			}
		} else {
			const scriptResult = await this._executeMetaScript('', args, metaConfig);
			let subProcessResult;
			if (this._meta.sub && Object.keys(this._meta.sub).length > 1) {
				subProcessResult = await this._executeSubProcess(scriptResult.type, scriptResult.payload);
			} else {
				subProcessResult = scriptResult;
			}
			result = await this._executeNextProcess(subProcessResult.type, subProcessResult.payload);
		}

		return result;
	}

	async _executeMetaScript(type, payload, meta) {
		let result = {
			type, payload
		};

		if (this._meta) {
			let metaConfig = this._meta;
			if (metaConfig) {
				this.updateProps(this._config.props = {}, payload, this._config._originProps, this._meta.props);
				if (metaConfig.script) {
					let func;
					if (metaConfig.script.indexOf(linkScheme) === 0) {
						func = this._vm.getScript(metaConfig.script.replace(linkScheme, ''));
					} else {
						func = new Function('args', 'props', 'target', 'global', 'vm', 'scope', warpAsyncScript(metaConfig.script));
					}
					if (func) {
						let globalContext = this._vm.globalContext;
						globalContext.gameStage = engine.gameStage;
						globalContext.dataCenter = engine.gameStage.dataCenter;
						globalContext.env = engine.env;
						result = await func(payload, this._config.props, this._target, globalContext, this._vm, this._parent.scope);
						logProcess(metaConfig, this._vm, this, `output: <${result.type}>`, result.payload);
					} else {
						console.warn('script lose:', metaConfig.script);
					}
				}
			} else {
				console.warn(`process meta [${meta}] not found`)
			}
		}

		return result;
	}

	async _executeSubProcess(type, payload) {
		let result = {
			type, payload
		};
		if (this._meta) {
			let {sub, subEntry} = this._meta;
			if (sub) {
				result = await this._vm.executeProcess(objClone(sub), subEntry, this, payload);
			}
		}

		return result;
	}

	async _executeNextProcess(type, payload) {
		let result = {type, payload};
		if (this._config.output) {
			const {output} = this._config;
			if (output.hasOwnProperty(type)) {
				const nextPids = output[type];
				if (nextPids.length > 0) {
					result = await this._vm.executeProcess(this._sequence, nextPids[0], this.parent, payload);
				}
			}
		}

		return result;
	}

	/**
	 * 获取过程配置
	 * @param id
	 * @return {null}
	 */
	getProcessMeta(id) {
		let meta;
		if (this._meta && this._meta.metas) { //如果有内联过程
			for (let temp of this._meta.metas) {
				if (temp.id === id) {
					meta = temp;
					break;
				}
			}
		}
		if (!meta) {
			meta = this._parent ? this._parent.getProcessMeta(id) : null;
		}
		if (!meta) {
			meta = this._vm.getMeta(id);
		}
		return meta;
	}

	/**
	 * 获取props
	 */
	getProps(key?) {
		if (key && this._config.props) {
			return this._config.props[key];
		}
		return this._config.props;
	}

	/**
	 * 更新props
	 */
	updateProps(props, args, originProps, propsConfig) {
		if (originProps) {
			for (let key in propsConfig) {
				let value = originProps[key];
				const valueType = typeof value;
				if (valueType == 'object') {
					let name = value.value;
					switch (value.type) {
						case 'link':
							let linkedValue = this.resolveLinkedProp(value, key);
							if (linkedValue !== undefined) {
								props[key] = linkedValue;
							} else {
								props[key] = undefined;
							}
							break;
						case 'static':
							props[key] = name;
							break;
						case 'scope':
							props[key] = getDataByPath(this._parent.scope, name);
							break;
						case 'arguments':
							props[key] = args ? getDataByPath(args, name) : undefined;
							break;
						case 'data-center':
							let nameValue = dataCenter.getDataByName(name);
							props[key] = nameValue !== undefined ? nameValue : dataCenter.getDataByPath(name);
							break;
						case 'env':
							props[key] = getDataByPath(env, name);
							break;
						case 'map':
							this.updateProps(props[key] = {}, args, name, name);
							break;
					}
				} else if (value && value.indexOf && value.indexOf(nodeScheme) === 0) {
					let uuid = value.replace(nodeScheme, '');
					if (uuid) {
						props[key] = this._vm.globalContext.gameStage.findChildByUUID(uuid);
					}
				} else if (originProps[key] !== undefined) {
					props[key] = originProps[key];
				}

				if (props[key] === undefined && propsConfig[key].hasOwnProperty('default')) {
					props[key] = propsConfig[key]['default'];
				}
			}
		}
	}

	resolveLinkedProp(data, key) {
		let linkedKey = data.alias || key;
		if (this._parent) {
			return this._parent.getProps(linkedKey);
		}
	}
}

function warpAsyncScript(source) {
	return `return new Promise(function(resolve, reject){
	${source}
	
	function next(type, payload){resolve({type: type, payload: payload})}
	});`;
}
